package ru.yandex.market.clickphite.dictionary.loaders;

import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import ru.yandex.market.clickphite.dictionary.Dictionary;
import ru.yandex.market.clickphite.dictionary.DictionaryLoader;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.concurrent.TimeUnit;

/**
 * @author Denis Khurtin <dkhurtin@yandex-team.ru>
 */
public abstract class HttpDictionaryLoader implements DictionaryLoader {

    private static final int SOCKET_TIMEOUT = (int) TimeUnit.MINUTES.toMillis(5);

    public void load(Dictionary dictionary, DictionaryDataReader processor) throws IOException {
        try (CloseableHttpClient httpClient = HttpClients.createDefault()) {
            HttpGet httpGet = createGetRequest(dictionary);
            try (CloseableHttpResponse response = httpClient.execute(httpGet)) {
                if (response.getStatusLine().getStatusCode() != HttpStatus.SC_OK) {
                    throw new IOException("Wrong http status: " + response.getStatusLine().toString() +
                        " from " + httpGet);
                }
                HttpEntity entity = response.getEntity();
                if (entity == null) {
                    throw new IOException("Can't read from url " + httpGet);
                }
                processor.process(new BufferedReader(new InputStreamReader(entity.getContent(), "UTF-8")));
            }
        }
    }

    public abstract String getDownloadUrl(Dictionary dictionary);

    protected HttpGet createGetRequest(Dictionary dictionary) {
        RequestConfig requestConfig = RequestConfig.custom()
            .setSocketTimeout(SOCKET_TIMEOUT)
            .setConnectTimeout(SOCKET_TIMEOUT)
            .build();
        HttpGet httpGet = new HttpGet(getDownloadUrl(dictionary));
        httpGet.setConfig(requestConfig);
        return httpGet;
    }
}
