package ru.yandex.market.clickphite.dictionary.loaders.host2dc;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.google.gson.Gson;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.market.clickphite.dictionary.Dictionary;
import ru.yandex.market.clickphite.dictionary.DictionaryLoader;
import ru.yandex.market.clickphite.dictionary.loaders.HttpDictionaryLoader;
import ru.yandex.market.clickphite.dictionary.processors.TsvDictionaryProcessor;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.google.common.base.Preconditions.checkArgument;

public class Host2DcDictionaryLoader implements DictionaryLoader {
    private static final Gson GSON = new Gson();

    private String golemAllHostWithDcUrl;
    private String conductorDataCentersUrl;
    private String conductorDataCenterHostsUrl;

    @Override
    public void load(Dictionary dictionary, DictionaryDataReader processor) throws IOException {
        checkArgument(dictionary.getTable().equals("host2dc"), "Unknown dictionary %s", dictionary.getTable());

        Collection<DictionaryData> golemData = getDataFromGolem(dictionary);
        Collection<DictionaryData> fullData = appendConductorData(dictionary, golemData);

        String fullDataTsv = toTsvFormat(fullData);

        processor.process(new BufferedReader(new StringReader(fullDataTsv)));
    }

    @VisibleForTesting
    String toTsvFormat(Collection<DictionaryData> fullData) {
        return fullData.stream()
            .map(data -> String.format("%s\t%s\t%s", data.getHost(), data.getDc(), data.getStage()))
            .collect(Collectors.joining("\n"));
    }

    private Collection<DictionaryData> getDataFromGolem(Dictionary dictionary) throws IOException {
        List<DictionaryData> golemData = new ArrayList<>();
        DictionaryDataReader processor = golemDataProcessor(dictionary, golemData);
        processData(golemAllHostWithDcUrl, dictionary, processor);

        return golemData;
    }

    @VisibleForTesting
    DictionaryDataReader golemDataProcessor(Dictionary dictionary, Collection<DictionaryData> golemData) {
        TsvDictionaryProcessor golemProcessor = new TsvDictionaryProcessor();
        return reader -> golemProcessor.insertData(dictionary, reader, golemDataConsumer(golemData));
    }

    private Consumer<List<Object>> golemDataConsumer(Collection<DictionaryData> dataList) {
        return data -> {
            String dc = (String) data.get(1);
            if (Strings.isNullOrEmpty(dc)) {
                return;
            }
            dataList.add(new DictionaryData((String) data.get(0), dc, (String) data.get(2)));
        };
    }

    private Collection<DictionaryData> appendConductorData(Dictionary dictionary, Collection<DictionaryData> golemData)
        throws IOException {

        List<ConductorDc> conductorDcs = new ArrayList<>();
        processData(conductorDataCentersUrl, dictionary, reader -> toConductorDc(reader, conductorDcs));

        Map<String, DictionaryData> hostToData = golemData.stream()
            .collect(Collectors.toMap(DictionaryData::getHost, Function.identity()));

        for (ConductorDc conductorDc : conductorDcs) {
            String dcName = conductorDc.getName();
            String dcDataUrl = conductorDataCenterHostsUrl + "/" + dcName;
            processData(dcDataUrl, dictionary, reader -> appendConductorDcData(reader, dcName, hostToData));
        }

        return hostToData.values();
    }

    @VisibleForTesting
    void appendConductorDcData(BufferedReader reader, String dc, Map<String, DictionaryData> hostToData) {
        reader.lines().forEach(host -> {
            if (Strings.isNullOrEmpty(host)) {
                return;
            }

            DictionaryData savedData = hostToData.get(host);
            if (savedData == null) {
                hostToData.put(host, new DictionaryData(host, dc, null));
            } else if (savedData.getDc().isEmpty()) {
                savedData.setDc(dc);
            }
        });
    }

    @VisibleForTesting
    void toConductorDc(BufferedReader reader, List<ConductorDc> data) {
        for (ConductorDc conductorDc : GSON.fromJson(reader, ConductorDc[].class)) {
            // filter only root dcs
            if (conductorDc.getParent() == null) {
                data.add(conductorDc);
            }
        }
    }

    private void processData(String dataUrl, Dictionary dictionary, DictionaryDataReader processor) throws IOException {
        HttpDictionaryLoader dictionaryLoader = new HttpDictionaryLoader() {
            @Override
            public String getDownloadUrl(Dictionary dictionary) {
                return dataUrl;
            }
        };

        dictionaryLoader.load(dictionary, processor);
    }

    @Required
    public void setGolemAllHostWithDcUrl(String golemAllHostWithDcUrl) {
        this.golemAllHostWithDcUrl = golemAllHostWithDcUrl;
    }

    @Required
    public void setConductorDataCentersUrl(String conductorDataCentersUrl) {
        this.conductorDataCentersUrl = conductorDataCentersUrl;
    }

    @Required
    public void setConductorDataCenterHostsUrl(String conductorDataCenterHostsUrl) {
        this.conductorDataCenterHostsUrl = conductorDataCenterHostsUrl;
    }
}
