package ru.yandex.market.clickphite.dictionary.processors;

import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

/**
 * @author Tatiana Litvinenko <a href="mailto:tanlit@yandex-team.ru"></a>
 * @date 22.09.2015
 */
public class CategoryTree {
    private static final Logger log = LogManager.getLogger();

    private static final String ID_FIELD = "id";
    private static final String PARENT_ID_FIELD = "parent_id";
    private static final String NAME_FIELD = "title";
    private static final String NOT_LEAF_FIELD = "not_leaf";
    private static final String PUBLISHED_FIELD = "published";

    private final Long2ObjectOpenHashMap<TreeNode> tree = new Long2ObjectOpenHashMap<>();

    public void addNode(Map<String, String> values) {
        try {
            int id = Integer.parseInt(values.get(ID_FIELD));
            TreeNode node = new TreeNode(
                id,
                Integer.parseInt(values.get(PARENT_ID_FIELD)),
                values.get(NAME_FIELD),
                "0".equals(values.get(NOT_LEAF_FIELD)),
                "1".equals(values.get(PUBLISHED_FIELD))
            );
            tree.put(id, node);
        } catch (Exception e) {
            log.warn("Can't parse category tree node: ", e);
        }
    }

    public void buildFullNodeNames() {
        for (TreeNode node : tree.values()) {
            buildFullNodeName(node);
        }
    }

    private void buildFullNodeName(TreeNode node) {
        StringBuilder sb = new StringBuilder(node.name);
        TreeNode parent = tree.get(node.parentId);
        while (parent != null && parent.id != parent.parentId) {
            sb.insert(0, parent.name + "\t");
            parent = tree.get(parent.parentId);
        }
        node.setFullName("\t" + sb + "\t");
    }

    public void consumeNodeValues(Consumer<List<Object>> consumer) {
        tree.values().stream().map(TreeNode::getValues).forEach(consumer);
    }

    public static class TreeNode {
        private final long id;
        private final long parentId;
        private final String name;
        private final boolean leaf;
        private final boolean published;
        private String fullName;

        public TreeNode(long id, long parentId, String name, boolean leaf, boolean published) {
            this.id = id;
            this.parentId = parentId;
            this.name = name;
            this.leaf = leaf;
            this.published = published;
        }

        public void setFullName(String fullName) {
            this.fullName = fullName;
        }

        public List<Object> getValues() {
            return Arrays.asList(id, parentId, name, fullName, leaf ? 1 : 0, published ? 1 : 0);
        }

        @Override
        public String toString() {
            return getValues().toString();
        }
    }
}
