package ru.yandex.market.clickphite.dictionary.processors;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.google.common.base.Preconditions;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.ColumnTypeBase;
import ru.yandex.market.clickphite.dictionary.Dictionary;
import ru.yandex.market.clickphite.dictionary.DictionaryProcessor;

import java.io.BufferedReader;
import java.io.IOException;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;

/**
 * @author Denis Khurtin <dkhurtin@yandex-team.ru>
 */
public class JsonDictionaryProcessor implements DictionaryProcessor {

    private static final ObjectMapper MAPPER = new ObjectMapper();

    @Override
    public void insertData(Dictionary dictionary, BufferedReader reader,
                           Consumer<List<Object>> consumer) throws IOException {
        String line;
        while ((line = reader.readLine()) != null) {
            JsonNode jsonNode = MAPPER.readTree(line);

            List<Column> columns = dictionary.getColumns();
            List<Object> values = new ArrayList<>(columns.size());

            for (Column column : columns) {
                JsonNode node = jsonNode.get(column.getName());
                values.add(node == null || node.isNull() ? null : parse(node, column.getType()));
            }

            consumer.accept(values);
        }
    }

    protected Object parse(JsonNode node, ColumnTypeBase type) {
        Preconditions.checkArgument(!Objects.requireNonNull(node).isNull());
        if (node.isArray()) {
            ArrayNode arrayNode = (ArrayNode) node;
            Preconditions.checkState(type instanceof ColumnType);
            ColumnType columnType = (ColumnType) type;

            Object[] array = new Object[arrayNode.size()];
            for (int i = 0; i < arrayNode.size(); ++i) {
                array[i] = parse(arrayNode.get(i), columnType.getSubtype());
            }
            return array;
        }

        DateFormat dateFormat = type == ColumnType.Date
                              ? ColumnTypeBase.dateFormatHolder.get()
                              : ColumnTypeBase.dateTimeFormatHolder.get();
        return type.parseValue(node.asText(), dateFormat);
    }
}
