package ru.yandex.market.clickphite.graphite;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.graphite.MetricValidator;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.io.PrintWriter;
import java.net.Socket;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 15/12/14
 */
public class GraphiteClient implements Runnable {

    private static final Logger log = LogManager.getLogger();
    private static final int MAX_WRONG_METRICS_REPORT = 20;
    private static final int WRONG_METRICS_REPORT_INTERVAL_MINUTES = 10;

    private List<String> hosts = Collections.singletonList("localhost");
    private int port = 42000;

    private static final int DEFAULT_SEND_LIMIT_PER_MIN = 10_000;
    private Random random = new Random();

    private final AtomicLong totalCount = new AtomicLong(0);
    private final AtomicLong minutesWorked = new AtomicLong(0);
    private final AtomicLong lastMinuteCount = new AtomicLong(0);
    private final AtomicLong currentMinuteCount = new AtomicLong(0);
    private final int sendLimitPerMinute;
    private Map<String, String> wrongMetricFieldNameToMetric = new ConcurrentHashMap<>();
    private Instant lastWrongMetricsReport = Instant.ofEpochSecond(0);

    private final ScheduledThreadPoolExecutor executor = new ScheduledThreadPoolExecutor(1);

    public GraphiteClient(int sendLimitPerMinute) {
        this.sendLimitPerMinute = sendLimitPerMinute;
        executor.scheduleAtFixedRate(this, 1, 1, TimeUnit.MINUTES);
    }

    public GraphiteClient() {
        this(DEFAULT_SEND_LIMIT_PER_MIN);
    }

    @Override
    public void run() {
        lastMinuteCount.set(currentMinuteCount.get());
        currentMinuteCount.set(0);
        minutesWorked.incrementAndGet();
        totalCount.addAndGet(lastMinuteCount.get());

        long minutesFromLastReport = Duration.between(lastWrongMetricsReport, Instant.now()).toMinutes();
        if (minutesFromLastReport >= WRONG_METRICS_REPORT_INTERVAL_MINUTES) {
            lastWrongMetricsReport = Instant.now();

            reportWrongMetrics();
            wrongMetricFieldNameToMetric.clear();
        }
    }

    private void reportWrongMetrics() {
        if (wrongMetricFieldNameToMetric.isEmpty()) {
            return;
        }

        String report = wrongMetricFieldNameToMetric.entrySet().stream()
            .map(e -> String.format("metric: %s", e.getValue()))
            .collect(Collectors.joining("\n"));

        log.warn("Wrong metric name report:\n" + report);
    }

    public void send(List<Metric> metrics) throws IOException {
        while (currentMinuteCount.get() > sendLimitPerMinute) {
            try {
                TimeUnit.SECONDS.sleep(1);
            } catch (InterruptedException e) {
                throw new InterruptedIOException(e.toString());
            }
        }

        currentMinuteCount.addAndGet(metrics.size());
        doSend(metrics);
    }

    public long getAverageMetricsPerMin() {
        if (minutesWorked.get() == 0) {
            return 0;
        }

        return totalCount.get() / minutesWorked.get();
    }

    public long getLastMinuteMetricsCount() {
        return lastMinuteCount.get();
    }

    public String getStatus() {
        return "Average metrics per min: " + getAverageMetricsPerMin()
            + ", last minute:" + getLastMinuteMetricsCount() + " metrics.";

    }

    private String selectHost() {
        if (hosts.size() == 1) {
            return hosts.get(0);
        }
        return hosts.get(random.nextInt(hosts.size()));
    }

    private void doSend(List<Metric> metrics) throws IOException {
        Socket socket = createSocket();
        PrintWriter writer = new PrintWriter(socket.getOutputStream());

        for (Metric metric : metrics) {
            String metricNameString = metric.getName();
            if (!MetricValidator.DEFAULT.validate(metricNameString)) {
                if (wrongMetricFieldNameToMetric.size() < MAX_WRONG_METRICS_REPORT) {
                    wrongMetricFieldNameToMetric.put(
                        metric.getFieldName() != null ? metric.getFieldName() : metricNameString, metricNameString
                    );
                }

                continue;
            }

            String metricLine = metricNameString + " " + metric.getValue() + " " + metric.getTimestampSeconds();
            writer.println(metricLine);
        }

        writer.close();
        socket.close();
    }

    protected Socket createSocket() throws IOException {
        return new Socket(selectHost(), port);
    }

    public void setHosts(String hosts) {
        this.hosts = Arrays.asList(hosts.split(","));
    }

    public String getHost() {
        return selectHost();
    }

    public void setPort(int port) {
        this.port = port;
    }

}
