package ru.yandex.market.clickphite.graphite;

import com.google.common.base.Preconditions;
import ru.yandex.market.clickphite.metric.GraphiteMetricContext;

import java.util.Arrays;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 15/12/14
 */
public class Metric {
    private static final String[] EMPTY_ARRAY = new String[0];

    private final int timestampSeconds;
    private final double value;

    private final String[] metricNameParts;
    private final String[] splitValues;
    private final String postfix;

    private final String fieldName;
    private int metricNameLength;

    /**
     * @param timestampSeconds
     * @param value
     * @param metricNamePartsTotalLength см. {@link GraphiteMetricContext#metricNamePartsTotalLength}. Нужно для того,
    +     *                              чтобы быстро вычислять место, необходимое под StringBuilder.
     * @param metricNameParts см. {@link GraphiteMetricContext#metricNameParts}.
     * @param splitValues
     * @param postfix
     * @param fieldName
     */
    public Metric(int timestampSeconds, double value,
                  int metricNamePartsTotalLength, String[] metricNameParts, String[] splitValues, String postfix,
                  String fieldName) {
        Preconditions.checkArgument(splitValues != null, "splitValues must be not null");

        // Такое условие получается потому, что между каждым значением в metricNameParts нужно вставить значение
        // сплита: metricNameParts = ["foo.", ".bar"] + splitValues = ["split_value"] -> "foo.split_value.bar"
        if (splitValues.length != metricNameParts.length - 1) {
            throw new IllegalArgumentException(
                String.format(
                    "metricNameParts must be less than splitValues by one, metricNameParts: %s, splitValues: %s",
                    Arrays.toString(metricNameParts), Arrays.toString(splitValues)
                )
            );
        }

        this.metricNameLength = metricNamePartsTotalLength + GraphiteMetricContext.getTotalStringsLength(splitValues);
        if (postfix != null) {
            this.metricNameLength += postfix.length();
        }

        this.timestampSeconds = timestampSeconds;
        this.metricNameParts = metricNameParts;
        this.splitValues = splitValues;
        this.postfix = postfix;
        this.value = value;
        this.fieldName = fieldName;
    }

    public Metric(int timestampSeconds, double value,
                  int metricNamePartsTotalLength, String[] metricNameParts, String[] splitValues) {
        this(
            timestampSeconds, value, metricNamePartsTotalLength, metricNameParts,
            splitValues, null, null
        );
    }

    public Metric(int timestampSeconds, double value, String simpleMetricName) {
        this(
            timestampSeconds, value, simpleMetricName.length(), new String[]{simpleMetricName},
            EMPTY_ARRAY, null, null
        );
    }

    public static String escapeName(String string) {
        if (string.equals("/")) {
            return "ROOT";
        }
        string = string.replace('.', '_').replace('/', '_').replace('\\', '_').replace('(', '_').replace(')', '_')
            .replace(' ', '_').replace('*', 'X').replace(':', '_').replace(';', '_').replace("!", "");
        if (string.charAt(0) == '_') {
            string = string.substring(1);
        }
        return string;
    }

    public int getTimestampSeconds() {
        return timestampSeconds;
    }

    public String getName() {
        // не хотим пересоздания массива, лежащего под StringBuilder, поэтому вычисляем его размер сразу
        final StringBuilder resultBuilder = new StringBuilder(metricNameLength);

        for (int i = 0; i < metricNameParts.length; ++i) {
            resultBuilder.append(metricNameParts[i]);

            if (i < splitValues.length) {
                resultBuilder.append(splitValues[i]);
            }
        }

        if (postfix != null) {
            resultBuilder.append(postfix);
        }

        return resultBuilder.toString();
    }

    @Override
    public String toString() {
        return "Metric{" +
            "timestampSeconds=" + timestampSeconds +
            ", name='" + getName() + '\'' +
            ", value=" + value +
            '}';
    }

    public double getValue() {
        return value;
    }

    public String getFieldName() {
        return fieldName;
    }
}
