package ru.yandex.market.clickphite.meta;

import com.google.common.base.Strings;
import com.google.common.collect.Range;
import com.google.common.collect.RangeSet;
import com.google.common.collect.TreeRangeSet;
import com.mongodb.BasicDBList;
import com.mongodb.MongoClient;
import com.mongodb.MongoClientOptions;
import com.mongodb.MongoClientURI;
import com.mongodb.ReadPreference;
import com.mongodb.WriteConcern;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.UpdateOneModel;
import com.mongodb.client.model.UpdateOptions;
import com.mongodb.client.model.WriteModel;
import org.bson.BsonReader;
import org.bson.BsonWriter;
import org.bson.Document;
import org.bson.codecs.Codec;
import org.bson.codecs.DecoderContext;
import org.bson.codecs.DocumentCodec;
import org.bson.codecs.EncoderContext;
import org.bson.codecs.configuration.CodecRegistries;
import org.bson.codecs.configuration.CodecRegistry;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.market.clickphite.metric.MetricContext;
import ru.yandex.market.clickphite.metric.MetricQueue;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 11/08/15
 */
public class ClickphiteMetaDao {
    private String mongoUrl;
    private String dbName = "health";
    private String metricCollectionName = "metrics";
    private int connectTimeoutMillis = (int) TimeUnit.SECONDS.toMillis(5);
    private int socketTimeoutMillis = (int) TimeUnit.MINUTES.toMillis(1);
    private boolean ssl;
    private String replicaSet;

    private MongoClient mongoClient;
    private MongoDatabase database;
    private MongoCollection<MetricData> metricCollection;


    private static final UpdateOptions UPSERT = new UpdateOptions().upsert(true);

    private static final CodecRegistry CODEC_REGISTRY = CodecRegistries.fromRegistries(
        MongoClient.getDefaultCodecRegistry(),
        CodecRegistries.fromCodecs(new MetricDataCodec())
    );

    private static final Codec<Document> DOCUMENT_CODEC = new DocumentCodec();

    public void afterPropertiesSet() throws Exception {
        MongoClientOptions.Builder options = MongoClientOptions.builder()
            .writeConcern(WriteConcern.MAJORITY)
            .readPreference(ReadPreference.primary())
            .connectTimeout(connectTimeoutMillis)
            .socketTimeout(socketTimeoutMillis)
            .sslEnabled(ssl);
        if (!Strings.isNullOrEmpty(replicaSet)) {
            options.requiredReplicaSetName(replicaSet);
        }
        mongoClient = new MongoClient(new MongoClientURI(mongoUrl, options));
        database = mongoClient.getDatabase(dbName).withCodecRegistry(CODEC_REGISTRY);
        metricCollection = database.getCollection(metricCollectionName, MetricData.class);
    }

    public void save(List<MetricContext> metricContexts) {
        if (metricContexts.isEmpty()) {
            return;
        }
        List<WriteModel<MetricData>> writeModels = new ArrayList<>();
        for (MetricContext metricContext : metricContexts) {
            MetricData metricData = metricContext.getMetricData();
            Document idDocument = toIdDocument(metricContext);
            writeModels.add(
                new UpdateOneModel<>(idDocument, new Document("$set", metricData), UPSERT)
            );
        }
        metricCollection.bulkWrite(writeModels);
    }

    public Document toIdDocument(MetricContext metricContext) {
        return new Document("_id", metricContext.getId());
    }

    public int load(List<MetricContext> metricContexts) {
        BasicDBList ids = new BasicDBList();
        for (MetricContext metricContext : metricContexts) {
            ids.add(metricContext.getId());
        }
        Document query = new Document("_id", new Document("$in", ids));
        Map<String, MetricData> metricDataById = new HashMap<>();
        for (MetricData metricData : metricCollection.find(query)) {
            metricDataById.put(metricData.getMetricId(), metricData);
        }
        int updatedCount = 0;
        for (MetricContext metricContext : metricContexts) {
            MetricData metricData = metricDataById.get(metricContext.getId());
            metricContext.setMetricData(metricData);
            if (metricData != null) {
                updatedCount++;
            }
        }
        return updatedCount;
    }

    @Required
    public void setMongoUrl(String mongoUrl) {
        this.mongoUrl = mongoUrl;
    }

    public void setDbName(String dbName) {
        this.dbName = dbName;
    }

    public void setMetricCollectionName(String metricCollectionName) {
        this.metricCollectionName = metricCollectionName;
    }

    private static class MetricDataCodec implements Codec<MetricData> {

        @Override
        public MetricData decode(BsonReader reader, DecoderContext decoderContext) {
            Document document = DOCUMENT_CODEC.decode(reader, decoderContext);
            return new MetricData(
                document.getString("id"),
                toMetricQueue(document.get("queue", Document.class))
            );
        }

        private MetricQueue toMetricQueue(Document document) {
            SortedMap<Long, RangeSet<Integer>> diff = new TreeMap<>();
            Document diffDocument = document.get("diff", Document.class);
            for (String key : diffDocument.keySet()) {
                Long timestampMillis = Long.parseLong(key);
                diff.put(timestampMillis, toRangeSet(diffDocument.get(key)));
            }

            return new MetricQueue(
                toRangeSet(document.get("mainRangeSet")),
                diff,
                document.getInteger("maxProcessedTimeSeconds", 0)
            );
        }

        @SuppressWarnings("unchecked")
        private RangeSet<Integer> toRangeSet(Object object) {
            List<Document> ranges = (List<Document>) object;
            RangeSet<Integer> rangeSet = TreeRangeSet.create();
            for (Document range : ranges) {
                rangeSet.add(Range.closedOpen(range.getInteger("start"), range.getInteger("end")));

            }
            return rangeSet;
        }

        @Override
        public void encode(BsonWriter writer, MetricData metricData, EncoderContext encoderContext) {
            Document document = new Document();
            document.put("id", metricData.getMetricId());
            document.put("queue", metricData.getMetricQueue().toDocument());
            document.put("updated", metricData.getUpdated());
            DOCUMENT_CODEC.encode(writer, document, encoderContext);
        }

        @Override
        public Class<MetricData> getEncoderClass() {
            return MetricData.class;
        }
    }

    public void setConnectTimeoutMillis(int connectTimeoutMillis) {
        this.connectTimeoutMillis = connectTimeoutMillis;
    }

    public void setSocketTimeoutMillis(int socketTimeoutMillis) {
        this.socketTimeoutMillis = socketTimeoutMillis;
    }

    public void setSsl(boolean ssl) {
        this.ssl = ssl;
    }

    public void setReplicaSet(String replicaSet) {
        this.replicaSet = replicaSet;
    }
}
