package ru.yandex.market.clickphite.metric;

import ru.yandex.common.util.collections.CollectionUtils;
import ru.yandex.market.clickphite.ClickHouseTable;
import ru.yandex.market.clickphite.DateTimeUtils;
import ru.yandex.market.clickphite.config.metric.GraphiteMetricConfig;
import ru.yandex.market.clickphite.config.metric.MetricField;
import ru.yandex.market.clickphite.dashboard.DashboardContext;
import ru.yandex.market.clickphite.graphite.Metric;
import ru.yandex.market.clickphite.monitoring.MonitoringContext;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 20/03/15
 */
public class GraphiteMetricContext extends MetricContext {
    private final String id;
    private final GraphiteMetricConfig metricConfig;
    private final Collection<DashboardContext> dashboardContexts;
    private final Collection<MonitoringContext> monitoringContexts;

    private static final int MAX_METRICS_BATCH_SIZE = 10_000;

    /**
     * Имя метрики, разбитое по сплитам. Пример:
     * "foo.${foo}.bar.${bar}" -> ["foo.", ".bar.", ""]
     *
     * Эти значения сохранены здесь для того, чтобы {@link Metric} занимала как можно меньше места.
     * Мы стараемся, чтобы {@link Metric} хранила только указатель на этот массив и значения сплитов в порядке,
     * соответствующем порядку сплитов в имени метрики.
     */
    private final String[] metricNameParts;

    /**
     * Суммарная длина всех строк в metricNameParts.
     */
    private final int metricNamePartsTotalLength;

    public GraphiteMetricContext(GraphiteMetricConfig metricConfig, ClickHouseTable clickHouseTable,
                                 Collection<DashboardContext> dashboardContexts,
                                 Collection<MonitoringContext> monitoringContexts) {
        super(metricConfig, clickHouseTable);

        this.dashboardContexts = dashboardContexts;
        this.monitoringContexts = monitoringContexts;

        String graphiteMetricName = metricConfig.getGraphiteName();

        id = "graphite." + graphiteMetricName;
        this.metricConfig = metricConfig;
        this.metricNameParts = StringTemplate.getConstantPartsFromString(graphiteMetricName);
        this.metricNamePartsTotalLength = getTotalStringsLength(metricNameParts);
    }

    public static int getTotalStringsLength(String[] values) {
        int sum = 0;
        for (int i = 0; i < values.length; i++) {
            sum += values[i].length();
        }
        return sum;
    }

    @Override
    public String getId() {
        return id;
    }

    @Override
    public SendStats sendMetrics(
        Iterable<MetricResultRow> resultRows, MetricServiceContext context
    ) throws IOException {
        List<Metric> metrics = new ArrayList<>();
        int sendCount = 0;
        for (MetricResultRow row : resultRows) {
            processMetricRow(row, metrics, context);
            if (metrics.size() >= MAX_METRICS_BATCH_SIZE) {
                sendCount += processMetricBatch(metrics, context);
            }
        }
        sendCount += processMetricBatch(metrics, context);
        return new SendStats(sendCount, 0);
    }

    private int processMetricBatch(List<Metric> metrics, MetricServiceContext context) throws IOException {
        context.sendGraphiteMetrics(metrics);
        notifyMonitorings(metrics);
        int count = metrics.size();
        metrics.clear();
        return count;
    }

    public void notifyMonitorings(List<Metric> metrics) {
        if (monitoringContexts != null && !monitoringContexts.isEmpty()) {
            for (MonitoringContext monitoringContext : monitoringContexts) {
                for (Metric metric : metrics) {
                    monitoringContext.onMetric(metric);
                }
            }
        }
    }

    public boolean hasDashboard() {
        return !CollectionUtils.isEmpty(dashboardContexts);
    }

    @Override
    public Collection<DashboardContext> getDashboardContexts() {
        return dashboardContexts != null ? dashboardContexts : Collections.emptyList();
    }

    @Override
    public Collection<MonitoringContext> getMonitoringContexts() {
        return monitoringContexts != null ? monitoringContexts : Collections.emptyList();
    }

    private void processMetricRow(MetricResultRow row, List<Metric> metrics, MetricServiceContext context) {
        List<? extends MetricField> splits = getSplits();
        String[] splitValues = new String[splits.size()];

        for (int i = 0; i < splits.size(); i++) {
            MetricField split = splits.get(i);

            String splitValue = row.getSplitValue(split.getClickHouseName());
            if (splitValue == null || splitValue.isEmpty()) {
                return;
            }
            splitValue = Metric.escapeName(splitValue);
            splitValues[i] = splitValue;
        }

        int timestampSeconds = row.getTimestampSeconds();
        if (getPeriod().isGraphiteOffset()) {
            timestampSeconds = DateTimeUtils.graphiteOffset(timestampSeconds);
        }

        if (hasDashboard()) {
            for (DashboardContext dashboardContext : getDashboardContexts()) {
                dashboardContext.notifyMetric(splitValues);
            }
        }

        switch (metricConfig.getType()) {
            case SIMPLE:
                double value = preprocessValue(row.getValue());
                if (Double.isNaN(value)) {
                    return;
                }
                metrics.add(
                    new Metric(timestampSeconds, value, metricNamePartsTotalLength, metricNameParts, splitValues)
                );
                break;
            case QUANTILE:
            case QUANTILE_TIMING:
            case QUANTILE_TIMING_WEIGHTED:
                double[] values = row.getQuantileValueArray();
                List<String> quantiles = metricConfig.getQuantiles();
                for (int i = 0; i < values.length; i++) {
                    values[i] = preprocessValue(values[i]);
                    if (Double.isNaN(values[i])) {
                        continue;
                    }
                    String quantile = quantiles.get(i);
                    // TODO: сейчас все постфиксы — это разные объекты, хотя в подавляеющем большинстве —
                    // одинаковые строки (производные от квантилей). Вероятно, можно завести мапу постфиксов,
                    // чтобы снизить нагрузку на GC
                    String postfix = "." + Metric.escapeName(quantile);
                    metrics.add(
                        new Metric(
                            timestampSeconds, values[i], metricNamePartsTotalLength, metricNameParts,
                            splitValues, postfix, quantile
                        )
                    );
                }
                break;
            default:
                throw new IllegalStateException();
        }
    }

    @Override
    public MetricStorage getStorage() {
        return MetricStorage.GRAPHITE;
    }
}
