package ru.yandex.market.clickphite.metric;

import com.google.common.annotations.VisibleForTesting;
import ru.yandex.market.clickphite.ClickHouseTable;
import ru.yandex.market.clickphite.config.metric.AbstractMetricConfig;
import ru.yandex.market.clickphite.config.metric.MetricField;
import ru.yandex.market.clickphite.config.metric.MetricPeriod;
import ru.yandex.market.clickphite.dashboard.DashboardContext;
import ru.yandex.market.clickphite.meta.MetricData;
import ru.yandex.market.clickphite.monitoring.MonitoringContext;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 23/12/14
 */
public abstract class MetricContext {
    protected final AbstractMetricConfig<?> metricConfig;
    private final ClickHouseTable clickHouseTable;
    private volatile MetricQueue metricQueue = new MetricQueue();
    private volatile long lastLoadTimeSeconds = -1;

    @VisibleForTesting
    protected MetricContext() {
        this.metricConfig = null;
        this.clickHouseTable = null;
    }

    public MetricContext(AbstractMetricConfig metricConfig, ClickHouseTable clickHouseTable) {
        this.metricConfig = metricConfig;
        this.clickHouseTable = clickHouseTable;
    }

    public MetricData getMetricData() {
        return new MetricData(getId(), metricQueue.copy());
    }

    public void setMetricData(MetricData metricData) {
        if (metricData != null) {
            metricQueue = metricData.getMetricQueue();
        } else {
            metricQueue = new MetricQueue();
        }
    }

    /**
     * @param resultRows
     * @param context
     * @return send metrics counts
     */
    public abstract SendStats sendMetrics(Iterable<MetricResultRow> resultRows,
                                          MetricServiceContext context) throws IOException;

    public abstract MetricStorage getStorage();

    public abstract String getId();

    public long getLastLoadTimeSeconds() {
        return lastLoadTimeSeconds;
    }

    public void setLastLoadTimeSeconds(long lastLoadTimeSeconds) {
        this.lastLoadTimeSeconds = lastLoadTimeSeconds;
    }

    public MetricPeriod getPeriod() {
        return metricConfig.getPeriod();
    }

    public List<? extends MetricField> getSplits() {
        return metricConfig.getSplits();
    }

    public AbstractMetricConfig<?> getMetricConfig() {
        return metricConfig;
    }

    public ClickHouseTable getClickHouseTable() {
        return clickHouseTable;
    }

    public Collection<DashboardContext> getDashboardContexts() {
        return Collections.emptyList();
    }

    public Collection<MonitoringContext> getMonitoringContexts() {
        return Collections.emptyList();
    }


    @Override
    public String toString() {
        return getId();
    }

    public MetricQueue getMetricQueue() {
        return metricQueue;
    }

    public double preprocessValue(double value) {
        return Double.isNaN(value) ? metricConfig.getValueOnNan() : value;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        MetricContext that = (MetricContext) o;

        String id = getId();
        String thatId = that.getId();
        return id != null ? id.equals(thatId) : thatId == null;
    }

    @Override
    public int hashCode() {
        String id = getId();
        return id != null ? id.hashCode() : 0;
    }


    public static class SendStats {
        private final long sentMetricsCount;
        private final long ignoredInvalidMetricsCount;

        public SendStats(long sentMetricsCount, long ignoredInvalidMetricsCount) {
            this.sentMetricsCount = sentMetricsCount;
            this.ignoredInvalidMetricsCount = ignoredInvalidMetricsCount;
        }

        public long getSentMetricsCount() {
            return sentMetricsCount;
        }

        public long getIgnoredInvalidMetricsCount() {
            return ignoredInvalidMetricsCount;
        }

        public SendStats plus(SendStats sendStats) {
            return new SendStats(
                this.sentMetricsCount + sendStats.sentMetricsCount,
                this.ignoredInvalidMetricsCount + sendStats.ignoredInvalidMetricsCount
            );
        }
    }
}
