package ru.yandex.market.clickphite.metric;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import ru.yandex.market.clickhouse.HttpResultRow;
import ru.yandex.market.clickphite.ClickHouseTable;
import ru.yandex.market.clickphite.ProcessStatus;
import ru.yandex.market.clickphite.QueryBuilder;
import ru.yandex.market.clickphite.config.metric.AbstractMetricConfig;
import ru.yandex.market.clickphite.config.metric.MetricField;
import ru.yandex.market.clickphite.config.metric.MetricPeriod;
import ru.yandex.market.clickphite.config.metric.SubAggregateConfig;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Метрики, значения которых могут быть запрошены одним запросом.
 */
public class MetricContextGroupImpl implements MetricContextGroup {
    private final String id;
    private final List<MetricContext> metricContexts;
    private final MetricPeriod period;
    private final ProcessStatus processStatus = new ProcessStatus();
    private final ClickHouseTable table;
    private final String filter;
    private final SubAggregateConfig subAggregate;
    private final MetricQueries metricQueries;
    private final List<? extends MetricField> splits;
    private final Map<MetricContext, Map<String, String>> splitsAlternativeNames = new HashMap<>();
    private final List<MetricStorage> storages;
    private final MetricContextGroup origin;
    private final int movingWindowPeriods;

    private MetricContextGroupImpl(List<MetricContext> metricContexts, MetricContextGroup origin) {

        MetricContext metricContext = metricContexts.get(0);
        AbstractMetricConfig<?> metricConfig = metricContext.getMetricConfig();

        this.metricContexts = metricContexts;
        this.id = metricContexts.stream().map(MetricContext::getId).collect(Collectors.joining(","));

        this.splits = metricConfig.getSplits();

        this.filter = metricConfig.getFilter();

        this.period = metricContext.getPeriod();
        this.table = metricContext.getClickHouseTable();
        this.subAggregate = metricConfig.getSubAggregate();
        this.movingWindowPeriods = metricConfig.getMovingWindowPeriods();
        this.metricQueries = QueryBuilder.buildQueries(this);
        this.storages = metricContexts.stream().map(MetricContext::getStorage).distinct().collect(Collectors.toList());
        this.origin = origin;


        Map<String, String> splitFieldToClickhouseName = splits.stream()
            .collect(Collectors.toMap(MetricField::getField, MetricField::getClickHouseName));

        for (int i = 1; i < metricContexts.size(); i++) {
            MetricContext context = metricContexts.get(i);
            Map<String, String> metricSplitNameMap = null;
            for (MetricField metricSplit : context.getSplits()) {
                String splitClickhouseName = splitFieldToClickhouseName.get(metricSplit.getField());
                Preconditions.checkNotNull(splitClickhouseName);

                String alternativeName = metricSplit.getClickHouseName();
                if (!splitClickhouseName.equals(alternativeName)) {
                    if (metricSplitNameMap == null) {
                        metricSplitNameMap = new HashMap<>();
                        splitsAlternativeNames.put(context, metricSplitNameMap);
                    }
                    metricSplitNameMap.put(alternativeName, splitClickhouseName);
                }
            }
        }
    }

    public static MetricContextGroup create(List<MetricContext> metricContexts) {
        return create(metricContexts, null);
    }

    public static MetricContextGroup create(List<MetricContext> metricContexts, MetricContextGroup origin) {
        Preconditions.checkArgument(!metricContexts.isEmpty());
        return new MetricContextGroupImpl(metricContexts, origin);
    }

    @Override
    public MetricPeriod getPeriod() {
        return period;
    }

    @Override
    public String toString() {
        return id;
    }

    @Override
    public ProcessStatus getProcessStatus() {
        return processStatus;
    }

    @Override
    public List<MetricContext> getMetricContexts() {
        return metricContexts;
    }

    @Override
    public ClickHouseTable getTable() {
        return table;
    }

    @Override
    public String getFilter() {
        return filter;
    }

    @Override
    public SubAggregateConfig getSubAggregate() {
        return subAggregate;
    }

    @Override
    public MetricQueries getQueries() {
        return getMetricQueries();
    }

    @Override
    public List<? extends MetricField> getSplits() {
        return splits;
    }

    @Override
    public List<AbstractMetricConfig<?>> getMetricConfigs() {
        return metricContexts.stream().map(MetricContext::getMetricConfig).collect(Collectors.toList());
    }

    @Override
    public SentMetricsStat.Builder sendMetrics(List<HttpResultRow> httpResultRows,
                                               MetricServiceContext metricServiceContext,
                                               SentMetricsStat.Builder statBuilder) throws IOException {
        for (int i = 0; i < getMetricContexts().size(); i++) {

            MetricContext metricContext = getMetricContexts().get(i);

            final int metricIndex = i;
            Stream<MetricResultRow> resultRows = httpResultRows.stream()
                .map(httpRow -> {
                    Map<String, String> splitNamesMap = this.splitsAlternativeNames.get(metricContext);
                    return new MetricResultRow(
                        httpRow, metricIndex, (splitNamesMap == null) ? Collections.EMPTY_MAP : splitNamesMap
                    );
                });

            Stopwatch stopwatch = Stopwatch.createStarted();
            MetricContext.SendStats stats = metricContext.sendMetrics(resultRows::iterator, metricServiceContext);
            statBuilder.addMetricSendDuration(metricContext, stopwatch.elapsed());
            statBuilder.addMetricSendStats(metricContext, stats);
        }

        return statBuilder;
    }

    @Override
    public Optional<MetricContextGroup> getOrigin() {
        return Optional.ofNullable(origin);
    }

    @Override
    public MetricQueries getMetricQueries() {
        return metricQueries;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        MetricContextGroupImpl that = (MetricContextGroupImpl) o;
        return Objects.equals(id, that.id);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id);
    }

    @Override
    public String getId() {
        return id;
    }

    @Override
    public int getMovingWindowPeriods() {
        return movingWindowPeriods;
    }
}
