package ru.yandex.market.clickphite.metric;

import org.apache.commons.lang3.time.DurationFormatUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.market.monitoring.ComplicatedMonitoring;
import ru.yandex.market.monitoring.MonitoringUnit;

import javax.annotation.concurrent.ThreadSafe;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author Anton Sukhonosenko <a href="mailto:algebraic@yandex-team.ru"></a>
 * @date 01.12.16
 */
@ThreadSafe
class MetricGroupMonitoring {
    private static final Logger log = LogManager.getLogger();
    public static final int FAILING_GROUPS_REPORT_INTERVAL_MINUTES = 10;

    private final Set<MetricContextGroup> allFailingGroups = new HashSet<>();

    private int failedMetricsMinutesToWarn = 15;
    private int failedMetricsMinutesToCrit = 30;

    private int massiveMetricFailurePercentToCrit = 5;
    private int massiveMetricFailureMinutesToCrit = 5;

    private volatile int totalMetricCount;

    private MonitoringUnit monitoringUnit;

    private Instant lastFailureReport = Instant.ofEpochSecond(0);

    public synchronized void reportSuccess(MetricContextGroup metricContextGroup) {
        allFailingGroups.remove(metricContextGroup);
        updateMonitoring();
    }

    public synchronized void reportFailure(MetricContextGroup metricContextGroup) {
        allFailingGroups.add(metricContextGroup);

        long minutesFromLastReport = Duration.between(lastFailureReport, Instant.now()).toMinutes();
        if (minutesFromLastReport >= FAILING_GROUPS_REPORT_INTERVAL_MINUTES) {
            lastFailureReport = Instant.now();

            log.error(
                "{} metric group(s) cannot be built: {}",
                allFailingGroups.size(),
                failingGroupsToString(allFailingGroups)
            );
        }

        updateMonitoring();
    }

    public synchronized void setTotalMetricCount(int totalMetricCount) {
        this.totalMetricCount = totalMetricCount;
    }

    public synchronized void setOk() {
        monitoringUnit.ok();
    }

    private void updateMonitoring() {
        Optional<String> massiveMetricFailureCrit = checkMassiveMetricFailure();
        if (massiveMetricFailureCrit.isPresent()) {
            monitoringUnit.critical(massiveMetricFailureCrit.get());
            return;
        }

        Optional<String> individualMetricFailureCrit = checkIndividualMetricFailure(failedMetricsMinutesToCrit);
        if (individualMetricFailureCrit.isPresent()) {
            monitoringUnit.critical(individualMetricFailureCrit.get());
            return;
        }

        Optional<String> individualMetricFailureWarn = checkIndividualMetricFailure(failedMetricsMinutesToWarn);
        if (individualMetricFailureWarn.isPresent()) {
            monitoringUnit.warning(individualMetricFailureWarn.get());
            return;
        }

        monitoringUnit.ok();
    }

    private List<MetricContextGroup> getGroupsThatWereFailingFor(int durationMinutes) {
        return allFailingGroups.stream()
            .filter(metricContextGroup ->
                getTimeSinceFirstErrorMillis(metricContextGroup) >= TimeUnit.MINUTES.toMillis(durationMinutes))
            .collect(Collectors.toList());
    }

    public synchronized void actualizeMetrics(List<MetricContextGroup> allMetricContextGroups) {
        allFailingGroups.retainAll(new HashSet<>(allMetricContextGroups));
        updateMonitoring();
    }

    public void setFailedMetricsMinutesToWarn(int failedMetricsMinutesToWarn) {
        this.failedMetricsMinutesToWarn = failedMetricsMinutesToWarn;
    }

    public void setFailedMetricsMinutesToCrit(int failedMetricsMinutesToCrit) {
        this.failedMetricsMinutesToCrit = failedMetricsMinutesToCrit;
    }

    public void setMassiveMetricFailurePercentToCrit(int massiveMetricFailurePercentToCrit) {
        this.massiveMetricFailurePercentToCrit = massiveMetricFailurePercentToCrit;
    }

    public void setMassiveMetricFailureMinutesToCrit(int massiveMetricFailureMinutesToCrit) {
        this.massiveMetricFailureMinutesToCrit = massiveMetricFailureMinutesToCrit;
    }

    @Required
    public void setMonitoring(ComplicatedMonitoring monitoring) {
        monitoringUnit = new MonitoringUnit("Metric monitoring");
        monitoring.addUnit(monitoringUnit);
    }

    private static String failingGroupsToString(Collection<MetricContextGroup> metricContextGroups) {
        return metricContextGroups.stream()
            .map(MetricGroupMonitoring::failingGroupToString)
            .collect(Collectors.joining(", "));
    }

    private static String failingGroupToString(MetricContextGroup metricContextGroup) {
        Duration timeSinceFirstError = Duration.of(getTimeSinceFirstErrorMillis(metricContextGroup), ChronoUnit.MILLIS);

        return String.format(
            "[%s] (first error: %s ago)",
            metricContextGroup.toString(),
            DurationFormatUtils.formatDuration(timeSinceFirstError.toMillis(), "HH 'hours' mm 'minutes'")
        );
    }

    private static long getTimeSinceFirstErrorMillis(MetricContextGroup metricContextGroup) {
        return System.currentTimeMillis() - metricContextGroup.getProcessStatus().getFirstErrorTimeInARowMillis();
    }


    private Optional<String> checkMassiveMetricFailure() {
        if (totalMetricCount == 0) {
            throw new IllegalStateException("Total metrics count must be set");
        }

        List<MetricContextGroup> failingGroups = getGroupsThatWereFailingFor(massiveMetricFailureMinutesToCrit);
        int failedMetricsPercent = failingGroups.size() * 100 / totalMetricCount;
        return failedMetricsPercent < massiveMetricFailurePercentToCrit
            ? Optional.empty()
            : Optional.of(formatMassiveFailureMessage(failingGroups, failedMetricsPercent));
    }

    private String formatMassiveFailureMessage(List<MetricContextGroup> failingGroups, int failedMetricsPercent) {
        return String.format(
            "More than %d%% (%d of %d -> %d%%) of metric groups cannot be built for more than %d minutes. " +
                "See clickphite.log for exceptions",
            massiveMetricFailurePercentToCrit,
            failingGroups.size(),
            totalMetricCount,
            failedMetricsPercent,
            massiveMetricFailureMinutesToCrit
        );
    }

    private Optional<String> checkIndividualMetricFailure(int minDurationMinutes) {
        if (minDurationMinutes < 0) {
            return Optional.empty();
        }

        List<MetricContextGroup> failingGroups = getGroupsThatWereFailingFor(minDurationMinutes);
        return failingGroups.isEmpty()
            ? Optional.empty()
            : Optional.of(formatIndividualMetricFailureMessage(failingGroups, minDurationMinutes));
    }

    private static String formatIndividualMetricFailureMessage(List<MetricContextGroup> failingGroups,
                                                               int minDurationMinutes) {
        return String.format(
            "%d metric group(s) cannot be built for more than %d minutes: %s",
            failingGroups.size(),
            minDurationMinutes,
            failingGroups.stream()
                .map(metricContextGroup -> "[" + metricContextGroup + "]")
                .collect(Collectors.joining(", "))
        );
    }
}
