package ru.yandex.market.clickphite.metric;

import com.google.common.collect.Range;
import com.google.common.collect.RangeSet;
import com.google.common.collect.TreeRangeSet;
import org.bson.Document;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 08/02/16
 */
public class MetricQueue {
    private final RangeSet<Integer> mainRangeSet;
    private final SortedMap<Long, RangeSet<Integer>> diff;
    private int maxProcessedTimeSeconds = 0;

    public MetricQueue() {
        mainRangeSet = TreeRangeSet.create();
        diff = new TreeMap<>();
    }

    public MetricQueue(RangeSet<Integer> mainRangeSet,
                       SortedMap<Long, RangeSet<Integer>> diff,
                       int maxProcessedTimeSeconds) {
        this.mainRangeSet = mainRangeSet;
        this.diff = diff;
        this.maxProcessedTimeSeconds = maxProcessedTimeSeconds;
    }

    public synchronized MetricQueue copy() {
        return new MetricQueue(TreeRangeSet.create(mainRangeSet), copy(diff), maxProcessedTimeSeconds);
    }

    private static SortedMap<Long, RangeSet<Integer>> copy(SortedMap<Long, RangeSet<Integer>> diff) {
        SortedMap<Long, RangeSet<Integer>> copy = new TreeMap<>();
        for (Map.Entry<Long, RangeSet<Integer>> originalEntry : diff.entrySet()) {
            copy.put(originalEntry.getKey(), TreeRangeSet.create(originalEntry.getValue()));
        }
        return copy;
    }

    public synchronized void add(Long timestampMillis, RangeSet<Integer> rangeSet) {
        RangeSet<Integer> existingRangeSet = diff.get(timestampMillis);
        if (existingRangeSet != null) {
            existingRangeSet.addAll(rangeSet);
        } else {
            diff.put(timestampMillis, TreeRangeSet.create(rangeSet));
        }
    }

    public synchronized void compact(long maxTimeMillis) {
        Iterator<Map.Entry<Long, RangeSet<Integer>>> iterator = diff.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<Long, RangeSet<Integer>> updateEntry = iterator.next();
            if (updateEntry.getKey() <= maxTimeMillis) {
                mainRangeSet.addAll(updateEntry.getValue());
                iterator.remove();
            }
        }
    }

    public synchronized void remove(long updateTimeMillis, RangeSet<Integer> rangeSet) {
        maxProcessedTimeSeconds = Math.max(maxProcessedTimeSeconds, rangeSet.span().upperEndpoint());

        mainRangeSet.removeAll(rangeSet);
        for (Map.Entry<Long, RangeSet<Integer>> updateEntry : diff.entrySet()) {
            if (updateEntry.getKey() <= updateTimeMillis) {
                updateEntry.getValue().removeAll(rangeSet);
            }
        }
    }

    public synchronized void remove(long updateTimeMillis, Range<Integer> range) {
        maxProcessedTimeSeconds = Math.max(maxProcessedTimeSeconds, range.upperEndpoint());
        mainRangeSet.remove(range);
        for (Map.Entry<Long, RangeSet<Integer>> updateEntry : diff.entrySet()) {
            if (updateEntry.getKey() <= updateTimeMillis) {
                updateEntry.getValue().remove(range);
            }
        }
    }

    public synchronized List<Range<Integer>> get(int maxTimeSeconds) {
        RangeSet<Integer> subRange;
        RangeSet<Integer> allRangeSet = TreeRangeSet.create(mainRangeSet);
        diff.values().forEach(allRangeSet::addAll);
        if (maxTimeSeconds > 0) {
            subRange = allRangeSet.subRangeSet(Range.lessThan(maxTimeSeconds));
        } else {
            subRange = allRangeSet;
        }
        List<Range<Integer>> ranges = new ArrayList<>(subRange.asRanges());
        Collections.reverse(ranges);
        return ranges;
    }

    public synchronized List<Range<Integer>> getAfterMaxProcessed(int maxTimeSeconds) {
        if (maxProcessedTimeSeconds == 0 || maxProcessedTimeSeconds > maxTimeSeconds) {
            return get(maxTimeSeconds);
        }
        RangeSet<Integer> subRange;
        RangeSet<Integer> allRangeSet = TreeRangeSet.create(mainRangeSet);
        diff.values().forEach(allRangeSet::addAll);
        if (maxTimeSeconds > 0) {
            subRange = allRangeSet.subRangeSet(Range.open(maxProcessedTimeSeconds, maxTimeSeconds));
        } else {
            subRange = allRangeSet.subRangeSet(Range.greaterThan(maxProcessedTimeSeconds));
        }
        List<Range<Integer>> ranges = new ArrayList<>(subRange.asRanges());
        Collections.reverse(ranges);
        return ranges;
    }

    public synchronized List<Range<Integer>> compactAndGet(int maxTimeSeconds, long maxCompactionTimeMillis) {
        compact(maxCompactionTimeMillis);
        return get(maxTimeSeconds);
    }

    public synchronized boolean hasActualTasks(int maxTimeSeconds) {
        Range<Integer> range = Range.lessThan(maxTimeSeconds);
        if (isConnected(mainRangeSet, range)) {
            return true;
        }
        return false;
    }

    private boolean isConnected(RangeSet<Integer> rangeSet, Range<Integer> range) {
        return !rangeSet.isEmpty() && rangeSet.span().isConnected(range);
    }

    public synchronized Document toDocument() {
        Document document = new Document();
        document.put("mainRangeSet", toRangeList(mainRangeSet));

        Document diffDocument = new Document();
        for (Map.Entry<Long, RangeSet<Integer>> diffEntry : diff.entrySet()) {
            diffDocument.put(diffEntry.getKey().toString(), toRangeList(diffEntry.getValue()));
        }
        document.put("diff", diffDocument);
        document.put("maxProcessedTimeSeconds", maxProcessedTimeSeconds);
        return document;
    }


    private List<Document> toRangeList(RangeSet<Integer> rangeSet) {
        List<Document> ranges = new ArrayList<>();
        for (Range<Integer> range : rangeSet.asRanges()) {
            Document rangeDocument = new Document();
            rangeDocument.put("start", range.lowerEndpoint());
            rangeDocument.put("end", range.upperEndpoint());
            ranges.add(rangeDocument);
        }
        return ranges;
    }

}
