package ru.yandex.market.clickphite.metric;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.hash.Hashing;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.common.util.date.DateUtil;
import ru.yandex.market.clickhouse.ClickhouseTemplate;
import ru.yandex.market.clickhouse.HttpResultRow;
import ru.yandex.market.clickphite.ClickHouseTable;
import ru.yandex.market.clickphite.DateTimeUtils;
import ru.yandex.market.clickphite.QueryBuilder;
import ru.yandex.market.clickphite.TimeRange;
import ru.yandex.market.clickphite.graphite.GraphiteClient;
import ru.yandex.market.clickphite.graphite.Metric;
import ru.yandex.market.clickphite.solomon.SolomonClient;
import ru.yandex.market.request.trace.TskvRecordBuilder;
import ru.yandex.market.statface.StatfaceClient;

import java.io.IOException;
import java.nio.charset.Charset;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 15/12/14
 */
public class MetricService implements MetricServiceContext {

    private static final Logger log = LogManager.getLogger();
    private static final Logger metricLog = LogManager.getLogger("metric");

    private static final DateUtil.ThreadLocalDateFormat LOG_DATE_FORMAT = new DateUtil.ThreadLocalDateFormat(
        "[dd/MMM/yyyy:HH:mm:ss Z]",
        Locale.ENGLISH
    );

    private static final long MAX_TIME_DIFFERENCE_MILLIS = TimeUnit.SECONDS.toMillis(5);
    private static final Joiner COMMA_JOINER = Joiner.on(",");

    private ClickhouseTemplate lightQueriesClickhouseTemplate;
    private ClickhouseTemplate mediumQueriesClickhouseTemplate;
    private ClickhouseTemplate heavyQueriesClickhouseTemplate;
    private GraphiteClient graphiteClient;
    private StatfaceClient statfaceClient;
    private SolomonClient solomonClient;

    private int warnRowsPerPeriod = 2000;
    private int maxRowsPerPeriod = 5000;

    private int maxConcurrentQueries = 10;
    private final Semaphore queriesSemaphore = new Semaphore(0, true);

    public void afterPropertiesSet() throws Exception {
        checkTimeDifference();
        queriesSemaphore.release(maxConcurrentQueries);
    }

    public int getTableFirstDataTimestampSeconds(ClickHouseTable table) {
        return lightQueriesClickhouseTemplate.queryForInt(
            "SELECT min(timestamp) FROM " + table.getFullName()
        );
    }

    /**
     * Проверяем, что в java и в Clickhouse одинаковое время, не больше чем MAX_TIME_DIFFERENCE_MILLIS,
     * а самое главное - одинаковый часовой пояс. Иначе начнеться кровькишкарасчленека
     */
    private void checkTimeDifference() {

        Date clickHouseDate = DateTimeUtils.dateFromTimeStampSeconds(
            lightQueriesClickhouseTemplate.queryForInt("SELECT toUnixTimestamp(now())")
        );
        Date javaDate = new Date();

        if (Math.abs(clickHouseDate.getTime() - javaDate.getTime()) > MAX_TIME_DIFFERENCE_MILLIS) {
            throw new RuntimeException(
                "Difference between java time and clickhouse time is more than " +
                    MAX_TIME_DIFFERENCE_MILLIS + "ms. Maybe problem with timezones. " +
                    "Java time " + javaDate + ", clickhouse time: " + clickHouseDate
            );
        }
    }

    public void updateMetricGroup(final MetricContextGroup metricContextGroup,
                                  final TimeRange timeRange, QueryWeight queryWeight) throws Exception {
        String query = QueryBuilder.placeTimeConditionToQuery(
            metricContextGroup.getQueries().getMainQuery(), timeRange, metricContextGroup.getMovingWindowPeriods()
        );

        final List<HttpResultRow> timestampResultRows = new ArrayList<>();

        final AtomicInteger currentTimestamp = new AtomicInteger(-1);
        final AtomicBoolean overflowTimestamp = new AtomicBoolean(false);

        final AtomicInteger processedRows = new AtomicInteger(0);
        final AtomicInteger ignoredRows = new AtomicInteger(0);

        queriesSemaphore.acquire();

        final SentMetricsStat.Builder statBuilder = SentMetricsStat.builder();

        long queryStartTimeMillis = System.currentTimeMillis();
        try {
            getClickhouseTemplate(queryWeight).query(
                query,
                rs -> {
                    int rowTimestamp = getRowTimestampSeconds(rs);
                    if (currentTimestamp.get() != rowTimestamp) {
                        warnMetricPerPeriod(timestampResultRows, metricContextGroup, currentTimestamp.get());

                        if (currentTimestamp.get() != -1) {
                            metricContextGroup.sendMetrics(timestampResultRows, this, statBuilder);
                        }

                        timestampResultRows.clear();
                        currentTimestamp.set(rowTimestamp);
                        overflowTimestamp.set(false);
                    }
                    if (overflowTimestamp.get()) {
                        ignoredRows.incrementAndGet();
                        return;
                    }
                    processedRows.incrementAndGet();
                    timestampResultRows.add(rs);

                    if (timestampResultRows.size() >= maxRowsPerPeriod) {
                        log.warn(
                            "Large amount of metrics (" + timestampResultRows.size() + " >= " + maxRowsPerPeriod +
                                "). IGNORING." +
                                "\nMetric: " + metricContextGroup +
                                "\ntimestamp: " + currentTimestamp.get() +
                                "\nquery: " + query
                        );
                        timestampResultRows.clear();
                        overflowTimestamp.set(true);
                    }
                },
                getQueryId(metricContextGroup)
            );
        } finally {
            queriesSemaphore.release();
        }

        warnMetricPerPeriod(timestampResultRows, metricContextGroup, currentTimestamp.get());
        metricContextGroup.sendMetrics(timestampResultRows, this, statBuilder);

        long queryTimeMillis = System.currentTimeMillis() - queryStartTimeMillis;

        logQuery(
            metricContextGroup, timeRange, queryTimeMillis, processedRows.get(), ignoredRows.get(), statBuilder.build(),
            queryWeight
        );
    }

    private ClickhouseTemplate getClickhouseTemplate(QueryWeight queryWeight) {
        switch (queryWeight) {
            case LIGHT:
                return lightQueriesClickhouseTemplate;
            case MEDIUM:
                return mediumQueriesClickhouseTemplate;
            case HEAVY:
                return heavyQueriesClickhouseTemplate;
            default:
                throw new UnsupportedOperationException("Unknown query weight: " + queryWeight);
        }
    }

    private String getQueryId(MetricContextGroup metricContextGroup) {
        String metricIds = metricContextGroup.getMetricContexts().stream()
            .map(this::getQueryId)
            .collect(Collectors.joining(","));

        return Hashing.md5().hashString(metricIds, Charset.defaultCharset()).toString();
    }

    private String getQueryId(MetricContext metricContext) {
        return metricContext.getId().replace('/', '-');
    }

    @Override
    public int getRowTimestampSeconds(HttpResultRow row) {
        return row.getInt(QueryBuilder.TIMESTAMP_INDEX);
    }


    private void warnMetricPerPeriod(List<HttpResultRow> timestampResultRows, MetricContextGroup metricContextGroup,
                                     int ts) {
        if (timestampResultRows.size() > warnRowsPerPeriod) {
            log.warn(
                "Large amount of result rows (" + timestampResultRows.size() +
                    "). Metric: " + metricContextGroup + " timestamp: " + ts
            );
        }
    }

    public void sendGraphiteMetrics(List<Metric> metrics) throws IOException {
        if (metrics.isEmpty()) {
            return;
        }
        graphiteClient.send(metrics);
    }

    private void logQuery(MetricContextGroup metricContextGroup, TimeRange timeRange,
                          long queryTimeMillis, int rowsRead, int rowsIgnored,
                          SentMetricsStat sentMetricsStat, QueryWeight queryWeight) {
        String logLine = getQueryLogLine(
            new Date(), metricContextGroup, timeRange, queryTimeMillis, rowsRead, rowsIgnored, sentMetricsStat,
            queryWeight
        );

        metricLog.info(logLine);
    }

    @VisibleForTesting
    @SuppressWarnings("checkstyle:parameternumber")
    static String getQueryLogLine(Date date, MetricContextGroup metricContextGroup, TimeRange timeRange,
                                  long queryTimeMillis, int rowsRead, int rowsIgnored,
                                  SentMetricsStat sentMetricsStat, QueryWeight queryWeight) {
        List<String> metricIds = new ArrayList<>();
        List<String> storages = new ArrayList<>();
        List<Integer> sendTimes = new ArrayList<>();
        List<Long> sentMetricsCount = new ArrayList<>();
        List<Long> invalidRowsIgnoredCounts = new ArrayList<>();

        for (MetricContext metricContext : metricContextGroup.getMetricContexts()) {
            Duration sendDuration = sentMetricsStat.getSendDuration(metricContext);
            MetricContext.SendStats sendStats = sentMetricsStat.getSendStats(metricContext);

            metricIds.add(metricContext.getId());
            storages.add(metricContext.getStorage().toString());
            sendTimes.add((int) sendDuration.toMillis());
            sentMetricsCount.add(sendStats.getSentMetricsCount());
            invalidRowsIgnoredCounts.add(sendStats.getIgnoredInvalidMetricsCount());
        }

        TskvRecordBuilder tskvBuilder = new TskvRecordBuilder();
        tskvBuilder.add("date", LOG_DATE_FORMAT.format(date));
        tskvBuilder.add("table", metricContextGroup.getTable().getFullName());
        tskvBuilder.add("period", metricContextGroup.getPeriod().name());
        tskvBuilder.add("metric_ids", COMMA_JOINER.join(metricIds));
        tskvBuilder.add("storage_per_id", COMMA_JOINER.join(storages));
        tskvBuilder.add("send_time_millis_per_id", COMMA_JOINER.join(sendTimes));
        tskvBuilder.add("metrics_sent_per_id", COMMA_JOINER.join(sentMetricsCount));
        tskvBuilder.add("start_timestamp_seconds", timeRange.getStartTimestampSeconds());
        tskvBuilder.add("end_timestamp_seconds", timeRange.getEndTimestampSeconds());
        tskvBuilder.add("query_time_millis", queryTimeMillis);
        tskvBuilder.add("rows_read", rowsRead);
        tskvBuilder.add("rows_ignored", rowsIgnored);
        tskvBuilder.add("invalid_rows_ignored_per_id", COMMA_JOINER.join(invalidRowsIgnoredCounts));
        tskvBuilder.add(
            "total_metrics_count_in_group",
            metricContextGroup.getOrigin()
                .map(o -> o.getMetricContexts().size())
                .orElse(metricContextGroup.getMetricContexts().size())
        );
        tskvBuilder.add("query_weight", queryWeight);
        return tskvBuilder.build();
    }

    @Required
    public void setLightQueriesClickhouseTemplate(ClickhouseTemplate lightQueriesClickhouseTemplate) {
        this.lightQueriesClickhouseTemplate = lightQueriesClickhouseTemplate;
    }

    @Required
    public void setMediumQueriesClickhouseTemplate(ClickhouseTemplate mediumQueriesClickhouseTemplate) {
        this.mediumQueriesClickhouseTemplate = mediumQueriesClickhouseTemplate;
    }

    @Required
    public void setHeavyQueriesClickhouseTemplate(ClickhouseTemplate heavyQueriesClickhouseTemplate) {
        this.heavyQueriesClickhouseTemplate = heavyQueriesClickhouseTemplate;
    }

    @Required
    public void setGraphiteClient(GraphiteClient graphiteClient) {
        this.graphiteClient = graphiteClient;
    }

    @Required
    public void setStatfaceClient(StatfaceClient statfaceClient) {
        this.statfaceClient = statfaceClient;
    }

    @Required
    public void setSolomonClient(SolomonClient solomonClient) {
        this.solomonClient = solomonClient;
    }

    public GraphiteClient getGraphiteClient() {
        return graphiteClient;
    }

    public StatfaceClient getStatfaceClient() {
        return statfaceClient;
    }

    @Override
    public SolomonClient getSolomonClient() {
        return solomonClient;
    }

    public void setMaxConcurrentQueries(int maxConcurrentQueries) {
        this.maxConcurrentQueries = maxConcurrentQueries;
    }

    public void setWarnRowsPerPeriod(int warnRowsPerPeriod) {
        this.warnRowsPerPeriod = warnRowsPerPeriod;
    }

    public void setMaxRowsPerPeriod(int maxRowsPerPeriod) {
        this.maxRowsPerPeriod = maxRowsPerPeriod;
    }
}
