package ru.yandex.market.clickphite.metric;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.clickphite.ClickHouseTable;
import ru.yandex.market.clickphite.config.metric.MetricField;
import ru.yandex.market.clickphite.config.metric.StatfaceReportConfig;
import ru.yandex.market.statface.StatfaceData;
import ru.yandex.market.statface.StatfaceReportConfiguration;
import ru.yandex.market.statface.StatfaceRow;

import java.io.IOException;
import java.util.Date;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 20/03/15
 */
public class StatfaceReportContext extends MetricContext {

    private static final Logger log = LogManager.getLogger();

    private final StatfaceReportConfig reportConfig;
    private final StatfaceReportConfiguration reportConfiguration;
    private final String id;

    private volatile boolean newReport = true;

    public StatfaceReportContext(StatfaceReportConfig reportConfig, ClickHouseTable clickHouseTable) {
        super(reportConfig, clickHouseTable);

        this.reportConfig = reportConfig;
        id = "statface/" + reportConfig.getReport() + "/" + reportConfig.getPeriod().getStatfacePeriod().name();
        reportConfiguration = reportConfig.createStatfaceReportConfiguration();
    }

    @Override
    public SendStats sendMetrics(
        Iterable<MetricResultRow> resultRows, MetricServiceContext context
    ) throws IOException {
        StatfaceData data = new StatfaceData(reportConfiguration);
        for (MetricResultRow resultRow : resultRows) {
            StatfaceRow row = createStatfaceRow(resultRow, context);
            if (row != null) {
                data.addRow(row);
            }
        }
        if (data.isEmpty()) {
            return new SendStats(0, 0);
        }
        if (newReport) {
            log.info("Creating statface report: " + reportConfiguration.getName());
            context.getStatfaceClient().createReport(reportConfiguration);
            newReport = false;
        }
        context.getStatfaceClient().sendData(data);
        return new SendStats(data.getRowCount(), 0);
    }

    public StatfaceRow createStatfaceRow(MetricResultRow resultRow, MetricServiceContext context) {
        Date date = context.getRowDate(resultRow);
        if (isTimeShift(date)) {
            return null;
        }
        StatfaceRow statfaceRow = new StatfaceRow(date);

        for (StatfaceReportConfig.Split split : reportConfig.getSplits()) {
            String splitValue = resultRow.getSplitValue(split.getClickHouseName());
            if (splitValue.isEmpty()) {
                return null;
            }
            if (split.isTree()) {
                splitValue = splitValue.replace("\\t", "\t");
            }
            statfaceRow.addDimension(split.getName(), splitValue);
        }
        int nonNanValues = 0;
        for (MetricField field : reportConfig.getFields()) {
            double value = preprocessValue(resultRow.getValue(field.getClickHouseName()));
            if (!Double.isNaN(value)) {
                nonNanValues++;
                statfaceRow.addValue(field.getName(), value);
            }
        }
        return nonNanValues > 0 ? statfaceRow : null;
    }

    //Долбанный перевод часов. Statface этого не любит.
    @SuppressWarnings("checkstyle:magicnumber")
    private boolean isTimeShift(Date date) {
        if (date.getTime() >= 1414270800000L && date.getTime() <= 1414281600000L) {
            switch (getPeriod()) {
                case ONE_SEC:
                case FIVE_SEC:
                case ONE_MIN:
                case FIVE_MIN:
                case HOUR:
                    return true;
                default:
                    return false;
            }
        }
        return false;
    }

    public StatfaceReportConfiguration getReportConfiguration() {
        return reportConfiguration;
    }

    @Override
    public MetricStorage getStorage() {
        return MetricStorage.STATFACE;
    }

    @Override
    public String getId() {
        return id;
    }

}
