package ru.yandex.market.clickphite.metric;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Этот класс умеет подставлять значения в строку вида "foo.${foo}.bar.${bar}".
 *
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 25.07.2018
 */
public class StringTemplate {
    private static final Pattern VARIABLE_PATTERN = Pattern.compile("\\$\\{([-_0-9a-zA-Z]+)\\}");

    // Строки между ${...}. Для строки "foo.${var1}.bar.${var2}" здесь будет ["foo.", ".bar.", ""].
    private final String[] constantParts;

    // Строки в ${...}. Для строки "foo.${var1}.bar.${var2}" здесь будет ["var1", "var2"]
    private final List<String> variables;

    private final boolean shouldThrowIfCapacityWasIncorrect;

    /**
     * @param stringWithSplits Строка с "${var}", например "foo.${foo}.bar.${bar}"
     */
    public StringTemplate(String stringWithSplits) {
        this(stringWithSplits, false);
    }

    /**
     * @param string Строка с "${var}", например "foo.${foo}.bar.${bar}"
     */
    @VisibleForTesting
    StringTemplate(String string, boolean shouldThrowIfCapacityWasIncorrect) {
        this.constantParts = getConstantPartsFromString(string);
        this.variables = getVariablesFromString(string);
        this.shouldThrowIfCapacityWasIncorrect = shouldThrowIfCapacityWasIncorrect;

        if (variables.size() != constantParts.length - 1) {
            throw new IllegalArgumentException(
                String.format(
                    "constantParts count must be less than variables count by one, constantParts: %s, variables: %s",
                    Arrays.toString(constantParts), variables
                )
            );
        }
    }

    public static List<String> getVariablesFromString(String string) {
        List<String> splits = new ArrayList<>();
        Matcher matcher = VARIABLE_PATTERN.matcher(string);
        while (matcher.find()) {
            splits.add(matcher.group(1));
        }
        return splits;
    }

    public static String[] getConstantPartsFromString(String string) {
        return VARIABLE_PATTERN.split(string, -1);
    }

    public boolean hasVariables() {
        return !variables.isEmpty();
    }

    public CharSequence render(Function<String, String> variableNameToValueFunction) {
        checkThatAllVariablesArePresent(variableNameToValueFunction);

        // Не хотим пересоздания массива, лежащего под StringBuilder, поэтому вычисляем его размер сразу
        int capacity = calculateCapacity(variableNameToValueFunction);
        StringBuilder result = new StringBuilder(capacity);

        for (int i = 0; i < variables.size(); ++i) {
            result.append(constantParts[i]);
            result.append(variableNameToValueFunction.apply(variables.get(i)));
        }
        result.append(constantParts[constantParts.length - 1]);

        if (shouldThrowIfCapacityWasIncorrect && (capacity != result.length())) {
            throw new IllegalStateException("Calculated capacity was wrong");
        }

        return result;
    }

    private void checkThatAllVariablesArePresent(Function<String, String> variableNameToValueMap) {
        for (String variable : variables) {
            Preconditions.checkArgument(
                variableNameToValueMap.apply(variable) != null,
                "Missing variable: {}",
                variable
            );
        }
    }

    private int calculateCapacity(Function<String, String> variableNameToValueMap) {
        int capacity = 0;
        for (String constantPart : constantParts) {
            capacity += constantPart.length();
        }
        for (String variable : variables) {
            capacity += variableNameToValueMap.apply(variable).length();
        }
        return capacity;
    }
}
