package ru.yandex.market.clickphite.metric.solomon;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import ru.yandex.market.clickphite.ClickHouseTable;
import ru.yandex.market.clickphite.config.metric.MetricSplit;
import ru.yandex.market.clickphite.config.metric.SolomonSensorConfig;
import ru.yandex.market.clickphite.metric.MetricContext;
import ru.yandex.market.clickphite.metric.MetricResultRow;
import ru.yandex.market.clickphite.metric.MetricServiceContext;
import ru.yandex.market.clickphite.metric.MetricStorage;
import ru.yandex.market.clickphite.solomon.SolomonShardId;
import ru.yandex.market.clickphite.solomon.SolomonShardNotFoundException;
import ru.yandex.market.clickphite.solomon.dto.SolomonPushRequestBody;
import ru.yandex.market.clickphite.solomon.dto.SolomonSensor;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 18.07.2018
 */
public class SolomonSensorContext extends MetricContext {
    private static final String QUANTILE_VARIABLE_NAME = "___quantile___";
    private static final int MAX_SENSORS_PER_REQUEST = 10000;

    private final String id;
    private final SolomonSensorIdTemplate sensorIdTemplate;
    private final boolean isQuantile;
    private final List<String> quantiles;
    private final Map<String, String> splitNameToClickHouseNameMap;

    public SolomonSensorContext(SolomonSensorConfig config, ClickHouseTable table) {
        super(config, table);
        this.id = createContextId(config);
        this.sensorIdTemplate = createSensorIdTemplate(config);
        this.isQuantile = config.getType().isQuantile();
        this.quantiles = config.getType().isQuantile() ? config.getQuantiles() : Collections.singletonList(null);
        this.splitNameToClickHouseNameMap = createSplitNameToClickHouseNameMap(config);
    }

    private static String createContextId(SolomonSensorConfig config) {
        // Нельзя использовать запятую как разделитель лейблов потому что в clickphite-metric.log запятая используется
        // как разделитель id метрик.
        return "solomon|" +
            config.getLabels().entrySet().stream()
                .sorted(Comparator.comparing(Map.Entry::getKey))
                .map(entry -> String.format("%s='%s'", entry.getKey(), entry.getValue()))
                .collect(Collectors.joining("|"));
    }

    private static SolomonSensorIdTemplate createSensorIdTemplate(SolomonSensorConfig config) {
        Map<String, String> labels = new HashMap<>(config.getLabels());
        if (config.getType().isQuantile()) {
            labels.put("quantile", "${" + QUANTILE_VARIABLE_NAME + "}");
        }
        return new SolomonSensorIdTemplate(labels);
    }

    private static Map<String, String> createSplitNameToClickHouseNameMap(SolomonSensorConfig config) {
        return config.getSplits().stream()
            .collect(Collectors.toMap(
                MetricSplit::getName,
                MetricSplit::getClickHouseName
            ));
    }

    /**
     * В ручку Соломона, в которую нужно передавать метрики, часть лейблов нужно передавать в урле, а часть в теле.
     * В урле передаётся id шарда (состоит из id проекта, сервиса и кластера), а в теле все остальные лейблы.
     * В конфигах Кликфита в id проекта, сервиса и кластера можно использовать значения сплитов.
     * Это значит что один SolomonSensorContext может пушить метрики для нескольких разных шардов.
     * Приходится дёргать ручку Соломона отдельно для каждого шарда.
     */
    @Override
    public SendStats sendMetrics(Iterable<MetricResultRow> resultRows, MetricServiceContext context) {
        List<PushSolomonSensorsRequest> requests = createPushRequestsFromResultRows(resultRows);
        long ignoredInvalidSensorsCount = 0;
        for (PushSolomonSensorsRequest request : requests) {
            try {
                long savedSensorsCount = context.getSolomonClient().push(
                    id,
                    request.getShardId(),
                    new SolomonPushRequestBody(
                        null,
                        request.getCommonLabels(),
                        request.getSensors()
                    )
                );
                Preconditions.checkState(
                    savedSensorsCount <= request.getSensors().size(),
                    "SolomonClient.push said that it saved more sensors than were given to it"
                );
                ignoredInvalidSensorsCount += request.getSensors().size() - savedSensorsCount;
            } catch (SolomonShardNotFoundException e) {
                // st/MARKETINFRA-3713 Создавать сервисы, кластеры и шарды автоматически
                //
                // Чтобы можно было пушить данные в Соломон, нужно сначала создать проект, сервис, кластер и шард.
                // Обычно это делается через UI, но у нас проектов, сервисов, кластеров и шардов много, удобнее
                // создавать автоматически.
                //
                // Если пуш упал с ошибкой "нет такого шарда", то создаём сервис, кластер и шард. Проект автоматически
                // не создаём, потому что не знаем кому на него дать права.
                //
                // Если этот контекст пушит в несколько шардов, то создаём сразу все, даже если упал только один шард,
                // потому что если создавать по одному, то это будет долго, и мониторинги могут успеть загореться.
                // Ничего страшного что можем попытаться создать уже существующие шарды, в
                // createServiceAndClusterAndShard такая ситуация считается успехом.
                requests.stream()
                    .map(PushSolomonSensorsRequest::getShardId)
                    .forEach(context.getSolomonClient()::createServiceAndClusterAndShard);

                // После создания шарда сразу не ретраим пуш, потому что шарды в Соломоне прорастают не сразу. Бросаем
                // исключение чтобы поретраить построение этой метрики.
                throw e;
            }
        }
        return new SendStats(
            requests.stream()
                .map(PushSolomonSensorsRequest::getSensors)
                .mapToInt(Collection::size)
                .sum(),
            ignoredInvalidSensorsCount
        );
    }

    @VisibleForTesting
    List<PushSolomonSensorsRequest> createPushRequestsFromResultRows(Iterable<MetricResultRow> resultRows) {
        return new PushSolomonSensorsRequestsBuilder(resultRows).getRequests();
    }

    @Override
    public MetricStorage getStorage() {
        return MetricStorage.SOLOMON;
    }

    @Override
    public String getId() {
        return id;
    }


    private class PushSolomonSensorsRequestsBuilder {
        private final Multimap<SolomonShardId, SolomonSensor> shardToSensorsMultimap = HashMultimap.create();

        PushSolomonSensorsRequestsBuilder(Iterable<MetricResultRow> rows) {
            for (MetricResultRow row : rows) {
                addSensors(row);
            }
        }

        List<PushSolomonSensorsRequest> getRequests() {
            return shardToSensorsMultimap.asMap().entrySet().stream()
                .flatMap(entry ->
                    // В Соломоне есть ограничение количества сенсоров на запрос. Если хотим передать больше сенсоров,
                    // то надо разбивать на несколько запросов.
                    Lists.partition(new ArrayList<>(entry.getValue()), MAX_SENSORS_PER_REQUEST).stream()
                        .map(sensorBatch ->
                            new PushSolomonSensorsRequest(
                                entry.getKey(),
                                sensorIdTemplate.getLabelsWithoutSplits(),
                                sensorBatch
                            )
                        )
                )
                .collect(Collectors.toList());
        }

        private void addSensors(MetricResultRow row) {
            double[] values = isQuantile ? row.getQuantileValueArray() : new double[]{row.getValue()};

            Preconditions.checkState(
                values.length == quantiles.size(),
                "quantile values count in row ({}) != quantiles count in config ({}) for sensor {}",
                values.length, quantiles.size(), id
            );

            SolomonShardId shardId = sensorIdTemplate.renderShardId(getSplitNameToValueMapFunction(row, null));

            for (int i = 0; i < values.length; i++) {
                double value = preprocessValue(values[i]);
                if (Double.isNaN(value) || Double.isInfinite(value)) {
                    continue;
                }
                shardToSensorsMultimap.put(
                    shardId,
                    new SolomonSensor(
                        sensorIdTemplate.renderLabelsWithSplits(getSplitNameToValueMapFunction(row, quantiles.get(i))),
                        SolomonSensor.Kind.DGAUGE,
                        value,
                        (long) row.getTimestampSeconds(),
                        null,
                        null
                    )
                );
            }
        }

        private Function<String, String> getSplitNameToValueMapFunction(MetricResultRow row, String quantile) {
            return splitName -> {
                if (QUANTILE_VARIABLE_NAME.equals(splitName)) {
                    return quantile;
                }
                return row.getSplitValue(splitNameToClickHouseNameMap.get(splitName));
            };
        }
    }
}
