package ru.yandex.market.clickphite.metric.solomon;

import com.google.common.base.Preconditions;
import ru.yandex.market.clickphite.metric.StringTemplate;
import ru.yandex.market.clickphite.solomon.SolomonShardId;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * В Соломоне id сенсора - это мапа лейблов, например '{project=market,cluster=stable,service=health,sensor=RPS}'.
 * В конфигах Кликфита это поле solomonSensors.labels. В лейблах можно использовать значения сплитов.
 *
 * При пуше сенсоров в Соломон лейблы приходится делить на три части:
 * 1. Лейблы project, service и cluster, которые все вместе формируют id шарда, передаются в урле. Это особенность API.
 *    Такие лейблы возвращает метод {@link #renderShardId}.
 * 2. Лейблы, в которых не используются сплиты, передаются в поле commonLabels. Значения этих лейблов одинаковы для
 *    всех сенсоров, которые передаёт один SolomonSensorContext. Их можно было бы передавать в поле sensors[].labels,
 *    но это создавало бы лишнюю нагрузку на GC и Соломон.
 *    Такие лейблы возвращает метод {@link #getLabelsWithoutSplits}
 * 3. Лейблы, в которых используются сплиты, передаются в поле sensors[].labels.
 *    Такие лейблы возвращает метод {@link #renderLabelsWithSplits}
 *
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 27.07.2018
 */
class SolomonSensorIdTemplate {
    private StringTemplate project;
    private StringTemplate service;
    private StringTemplate cluster;
    private final Map<String, String> labelsWithoutSplits = new HashMap<>();
    private final Map<String, StringTemplate> labelsWithSplits = new HashMap<>();

    @SuppressWarnings("MagicNumber")
    SolomonSensorIdTemplate(Map<String, String> allLabels) {
        allLabels.forEach(
            (name, value) -> {
                StringTemplate valueTemplate = new StringTemplate(value);
                if ("project".equals(name)) {
                    project = valueTemplate;
                    return;
                }
                if ("service".equals(name)) {
                    service = valueTemplate;
                    return;
                }
                if ("cluster".equals(name)) {
                    cluster = valueTemplate;
                    return;
                }
                if (valueTemplate.hasVariables()) {
                    labelsWithSplits.put(name, valueTemplate);
                } else {
                    labelsWithoutSplits.put(name, value);
                }
            }
        );

        Preconditions.checkState(
            labelsWithoutSplits.size() + labelsWithSplits.size() > 1,
            "There must be at least one more label apart from project, service, cluster and period"
        );

        // Костыль для обхода багофичи Соломона.
        // Push API Соломона требует чтобы у каждого сенсора было непустое поле labels, даже если есть commonLabels.
        // Обсуждение проблемы из чатика Соломона и примеры запросов: https://paste.yandex-team.ru/524923.
        // Значение поля labels генерится из labelsWithSplits, а commonLabels - из labelsWithoutSplits.
        // Чтобы обойти вышеописанную проблему, если мапа labelsWithSplits пуста, то перекладываем туда одно значение
        // из labelsWithoutSplits. Всегда перекладываем period, а не первый попавшийся лейбл, чтобы проще было писать
        // тесты (в тестах всегда точно знаем что должен переложиться period).
        if (labelsWithSplits.isEmpty()) {
            labelsWithSplits.put(
                "period",
                new StringTemplate(
                    Preconditions.checkNotNull(
                        labelsWithoutSplits.remove("period"),
                        "labelsWithoutSplits must contain label 'period'"
                    )
                )
            );
        }

        Preconditions.checkNotNull(project, "project");
        Preconditions.checkNotNull(service, "service");
        Preconditions.checkNotNull(cluster, "cluster");
        Preconditions.checkState(
            allLabels.size() == labelsWithoutSplits.size() + labelsWithSplits.size() + 3,
            "bug in SolomonSensorIdTemplate, some labels got lost or duplicated",
            allLabels.size(), labelsWithoutSplits.size(), labelsWithSplits.size()
        );
    }

    SolomonShardId renderShardId(Function<String, String> splitNameToValueMap) {
        return new SolomonShardId(
            project.render(splitNameToValueMap).toString(),
            service.render(splitNameToValueMap).toString(),
            cluster.render(splitNameToValueMap).toString()
        );
    }

    Map<String, String> getLabelsWithoutSplits() {
        return labelsWithoutSplits;
    }

    Map<String, CharSequence> renderLabelsWithSplits(Function<String, String> splitNameToValueMap) {
        return labelsWithSplits.entrySet().stream()
            .collect(Collectors.toMap(
                Map.Entry::getKey,
                entry -> entry.getValue().render(splitNameToValueMap)
            ));
    }
}
