package ru.yandex.market.clickphite.monitoring;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectRBTreeMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectSortedMap;
import ru.yandex.market.monitoring.MonitoringStatus;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 08/09/15
 */
public class DataPointList implements Iterable<DataPoint> {

    private final int size;
    private final int periodSeconds;
    private volatile int firstTsSeconds;
    private volatile int lastTsSeconds;

    /**
     * Можно реализовать эффективнее через масив с беганием по кругу,
     * но не стоит заниматься преждевременной оптимизацией.
     */
    private Int2ObjectSortedMap<DataPoint> dataPoints = new Int2ObjectRBTreeMap<>();

    public DataPointList(int size, int periodSeconds) {
        this.size = size;
        this.periodSeconds = periodSeconds;
    }

    @Override
    public Iterator<DataPoint> iterator() {
        return dataPoints.values().iterator();
    }

    public MonitoringStatusAndCause getStatus(int checkPoints, int warnsToCrit, int maxTimestampSeconds) {
        if (lastTsSeconds <= 0) {
            return null;
        }
        int processedPoints = 0;
        List<DataPoint> warnPoints = new ArrayList<>();

        int startTsSeconds = Math.min(maxTimestampSeconds / periodSeconds * periodSeconds, lastTsSeconds);

        for (int ts = startTsSeconds; ts >= firstTsSeconds && processedPoints < checkPoints; ts -= periodSeconds) {

            DataPoint point = dataPoints.get(ts);
            MonitoringStatus status = point.getStatus().getMonitoringStatus();
            if (status == MonitoringStatus.CRITICAL) {
                return new MonitoringStatusAndCause(MonitoringStatus.CRITICAL, Collections.singletonList(point));
            }
            if (processedPoints == 0 && !point.getStatus().hasMonitoringStatus()) {
                //Если у точки нету актуального статуса - не считаем нужным её считать.
                //Но если до этого уже были точки с данными - то ок.
                continue;
            }
            processedPoints++;
            if (status == MonitoringStatus.WARNING) {
                warnPoints.add(point);
                if (warnsToCrit > 0 && warnPoints.size() >= warnsToCrit) {
                    return new MonitoringStatusAndCause(MonitoringStatus.CRITICAL, warnPoints);
                }
            }
        }
        if (!warnPoints.isEmpty()) {
            return new MonitoringStatusAndCause(MonitoringStatus.WARNING, warnPoints);
        }
        return new MonitoringStatusAndCause(MonitoringStatus.OK, Collections.emptyList());
    }

    public DataPoint addValue(int timeStampSeconds, double value) {
        if (!validateTimeStamp(timeStampSeconds)) {
            return null;
        }

        DataPoint dataPoint = dataPoints.get(timeStampSeconds);
        if (dataPoint == null) {
            dataPoint = addDataPoint(timeStampSeconds);
        }

        dataPoint.setValue(value);

        return dataPoint;
    }

    private DataPoint addDataPoint(int timeStampSeconds) {
        if (lastTsSeconds <= 0) {
            lastTsSeconds = timeStampSeconds - size * periodSeconds;
        }
        for (int i = lastTsSeconds + periodSeconds; i < timeStampSeconds; i += periodSeconds) {
            dataPoints.put(i, new DataPoint(i));
        }
        DataPoint dataPoint = new DataPoint(timeStampSeconds);
        dataPoints.put(timeStampSeconds, dataPoint);
        lastTsSeconds = timeStampSeconds;
        firstTsSeconds = lastTsSeconds - (size - 1) * periodSeconds;
        clear();
        return dataPoint;
    }

    private void clear() {
        Iterator<Int2ObjectMap.Entry<DataPoint>> iterator = dataPoints.int2ObjectEntrySet().iterator();
        while (iterator.hasNext()) {
            Int2ObjectMap.Entry<DataPoint> entry = iterator.next();
            if (entry.getIntKey() >= firstTsSeconds) {
                return;
            }
            iterator.remove();
        }
    }

    private boolean validateTimeStamp(int timeStampSeconds) {
        if (timeStampSeconds % periodSeconds != 0) {
            throw new IllegalArgumentException("Invalid timestamp: " + timeStampSeconds);
        }
        if (timeStampSeconds < firstTsSeconds) {
            return false;
        }
        return true;
    }
}
