package ru.yandex.market.clickphite.monitoring;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.clickphite.DateTimeUtils;
import ru.yandex.market.clickphite.config.metric.GraphiteMetricConfig;
import ru.yandex.market.clickphite.config.monitoring.MonitoringConfig;
import ru.yandex.market.clickphite.graphite.Metric;
import ru.yandex.market.clickphite.metric.MetricListener;
import ru.yandex.market.monitoring.MonitoringStatus;

import java.net.URL;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 08/09/15
 */
public abstract class MonitoringContext implements MetricListener {

    private static final Logger log = LogManager.getLogger();

    protected final GraphiteMetricConfig metricConfig;
    protected final MonitoringConfig monitoringConfig;
    protected final MonitoringService monitoringService;

    private final String fieldName;


    private final DataPointList dataPoints;

    protected MonitoringStatusAndCause currentStatus;
    private MonitoringStatus lastNotifiedStatus;

    private int dataPointListSize = 120;

    public MonitoringContext(GraphiteMetricConfig metricConfig, MonitoringConfig monitoringConfig,
                             MonitoringService monitoringService, String fieldName) {
        this.metricConfig = metricConfig;
        this.monitoringConfig = monitoringConfig;
        this.monitoringService = monitoringService;
        this.fieldName = fieldName;
        this.dataPoints = new DataPointList(dataPointListSize, metricConfig.getPeriod().getDurationSeconds());
    }

    @Override
    public final void onMetric(Metric metric) {
        if (fieldName != null && !metric.getFieldName().equals(fieldName)) {
            return;
        }
        //TODO ignore old metrics
        double value = metric.getValue();
        DataPoint dataPoint;
        try {
            dataPoint = dataPoints.addValue(metric.getTimestampSeconds(), value);
        } catch (Exception e) {
            log.warn(
                String.format(
                    "Error during adding value for metric %s to data points: %s", metric.getName(), e.getMessage()
                )
            );
            return;
        }
        try {
            if (dataPoint != null) {
                onDataPoint(dataPoint);
                updateStatus();
            }
        } catch (Exception e) {
            log.error("Error in processing monitoring", e);
        }
    }

    public void updateStatus() {
        int maxTimestampSeconds = DateTimeUtils.currentTimeSeconds() - monitoringConfig.getMonitoringDelaySeconds();
        currentStatus = dataPoints.getStatus(
            monitoringConfig.getCheckPoints(), monitoringConfig.getWarnsToCrit(), maxTimestampSeconds
        );
    }


    public String getName() {
        return monitoringConfig.getName();
    }

    public String getMonitoringMessage() {
        String metricString = getMetricString();
        String causeString = getCauseString();

        StringBuilder builder = new StringBuilder();
        builder.append("Url: ").append(getMetricUrl()).append("\n");

        if (metricString != null) {
            builder.append("Metric: ").append(metricString);
        }
        if (causeString != null) {
            builder.append("Cause: ").append(causeString);
        }

        return builder.toString();
    }

    public URL getMetricUrl() {
        return monitoringService.createMetricUrl(metricConfig.getGraphiteName(), monitoringConfig.getQuantile());
    }


    public GraphiteMetricConfig getMetricConfig() {
        return metricConfig;
    }

    public MonitoringStatus getCurrentStatus() {
        if (currentStatus == null) {
            return null;
        }
        return currentStatus.getStatus();
    }

    public MonitoringStatus getLastNotifiedStatus() {
        return lastNotifiedStatus;
    }

    public void setLastNotifiedStatus(MonitoringStatus lastNotifiedStatus) {
        this.lastNotifiedStatus = lastNotifiedStatus;
    }

    public abstract MonitoringType getType();

    public abstract void onDataPoint(DataPoint dataPoint);

    public String getMetricString() {
        return null;
    }

    public String getCauseString() {
        return null;
    }
}
