package ru.yandex.market.clickphite.monitoring;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.market.calendar.YandexCalendarDay;
import ru.yandex.market.calendar.YandexCalendarService;
import ru.yandex.market.clickhouse.ClickhouseTemplate;
import ru.yandex.market.clickphite.ClickphiteService;
import ru.yandex.market.clickphite.config.ConfigurationService;
import ru.yandex.market.clickphite.graphite.GraphiteClient;
import ru.yandex.market.clickphite.graphite.Metric;
import ru.yandex.market.health.ProcessingQueue;
import ru.yandex.market.juggler.JugglerClient;
import ru.yandex.market.juggler.JugglerEvent;
import ru.yandex.market.monitoring.MonitoringStatus;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 08/09/15
 */
public class MonitoringService implements InitializingBean, Runnable {

    private static final Logger log = LogManager.getLogger();
    private static final Logger monitoringLog = LogManager.getLogger("monitoring");
    private static final int RU_REGION = 225;

    private int sleepTimeSeconds = 30;

    private int threadsCount = 10;

    private String monitoringUrlPath = "";

    private ClickhouseTemplate clickhouseTemplate;
    private YandexCalendarService yandexCalendarService;
    private GraphiteClient graphiteClient;
    private ConfigurationService configurationService;
    private ClickphiteService clickphiteService;

    private JugglerClient jugglerClient;
    private String monitoringCheckHost;

    private ExecutorService executorService;
    private ProcessingQueue<MonitoringContext> queue = new ProcessingQueue<MonitoringContext>() {
        @Override
        protected boolean isUpdated(MonitoringContext monitoringContext) {
            return true;
        }
    };

    @Override
    public void afterPropertiesSet() throws Exception {
        executorService = Executors.newFixedThreadPool(threadsCount);
        new Thread(this, "MonitoringService").start();
    }
//
//    public void submitTask(AsyncTask task) {
//        executorService.submit(new Runnable() {
//            @Override
//            public void run() {
//                task.run(MonitoringService.this);
//            }
//        });
//    }

    @Override
    public void run() {
        while (!Thread.interrupted()) {
            try {
                TimeUnit.SECONDS.sleep(sleepTimeSeconds);
                if (clickphiteService.isMaster()) {
                    doWork();
                }
            } catch (InterruptedException ignored) {
                break;
            } catch (Exception e) {
                log.error("ERROR in monitoring service", e);
                try {
                    TimeUnit.MINUTES.sleep(1);
                } catch (InterruptedException ignored) {
                }
            }
        }
    }

    private void doWork() {
        List<MonitoringContext> monitoringContexts = configurationService.getConfiguration().getMonitoringContexts();
        for (MonitoringContext monitoringContext : monitoringContexts) {
            try {
                processMonitoring(monitoringContext);
            } catch (Exception e) {
                log.error("Failed to process monitoring: " + monitoringContext.toString(), e);
            }
        }

        sendToJuggler(monitoringContexts);

        log.info("Processed " + monitoringContexts.size() + " monitorings");

    }

    private void sendToJuggler(List<MonitoringContext> monitorings) {
        List<JugglerEvent> events = monitorings.stream().
            filter(monitoring -> monitoring.getCurrentStatus() != null).
            map(this::toJugglerEvent).collect(Collectors.toList());
        try {
            jugglerClient.sendEvents(events);
        } catch (IOException e) {
            log.error("Failed to send events to juggler", e);
        }
    }

    private JugglerEvent toJugglerEvent(MonitoringContext monitoring) {
        return new JugglerEvent(
            monitoringCheckHost, monitoring.getName(),
            toJugglerStatus(monitoring.getCurrentStatus()), monitoring.getMonitoringMessage()
        );
    }

    private JugglerEvent.Status toJugglerStatus(MonitoringStatus status) {
        switch (status) {
            case OK:
                return JugglerEvent.Status.OK;
            case WARNING:
                return JugglerEvent.Status.WARN;
            case CRITICAL:
                return JugglerEvent.Status.CRIT;
            default:
                throw new IllegalStateException();
        }

    }

    private void processMonitoring(MonitoringContext context) throws IOException {
        MonitoringStatus currentStatus = context.getCurrentStatus();
        if (currentStatus == null) {
            return;
        }

        if (context.getLastNotifiedStatus() != currentStatus) {
            monitoringLog.info(context.getName() + ": " + currentStatus + " (" + context.getMonitoringMessage() + ")");
            context.setLastNotifiedStatus(currentStatus);
        }
    }

    public List<YandexCalendarDay> getSimilarDays(LocalDate date, int daysToSearch) throws IOException {
        List<YandexCalendarDay> days = yandexCalendarService.getDays(
            RU_REGION, date.minusDays(daysToSearch), date.plusDays(1)
        );

        YandexCalendarDay currentDay = days.get(days.size() - 2);
        YandexCalendarDay tomorrowDay = days.get(days.size() - 1);
        boolean isCurrentLastWorkDay = isLastWorkDay(currentDay, tomorrowDay);

        List<YandexCalendarDay> result = new ArrayList<>();
        for (int i = 0; i < days.size() - 2; i++) {
            YandexCalendarDay day = days.get(i);
            YandexCalendarDay nextDay = days.get(i + 1);
            boolean lastWorkDay = isLastWorkDay(day, nextDay);
            if (lastWorkDay && isCurrentLastWorkDay || day.isWorkday() == currentDay.isWorkday()) {
                result.add(day);
            }
        }
        return result;
    }

    private boolean isLastWorkDay(YandexCalendarDay day, YandexCalendarDay nextDay) {
        return day.isWorkday() && !nextDay.isWorkday();
    }


    public ClickhouseTemplate getClickhouseTemplate() {
        return clickhouseTemplate;
    }

    public GraphiteClient getGraphiteClient() {
        return graphiteClient;
    }

    @Required
    public void setMonitoringCheckHost(String monitoringCheckHost) {
        this.monitoringCheckHost = monitoringCheckHost;
    }

    @Required
    public void setClickhouseTemplate(ClickhouseTemplate clickhouseTemplate) {
        this.clickhouseTemplate = clickhouseTemplate;
    }

    @Required
    public void setYandexCalendarService(YandexCalendarService yandexCalendarService) {
        this.yandexCalendarService = yandexCalendarService;
    }

    @Required
    public void setConfigurationService(ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }

    @Required
    public void setGraphiteClient(GraphiteClient graphiteClient) {
        this.graphiteClient = graphiteClient;
    }

    @Required
    public void setJugglerClient(JugglerClient jugglerClient) {
        this.jugglerClient = jugglerClient;
    }

    @Required
    public void setMonitoringUrlPath(String monitoringUrlPath) {
        this.monitoringUrlPath = monitoringUrlPath;
    }

    @Required
    public void setClickphiteService(ClickphiteService clickphiteService) {
        this.clickphiteService = clickphiteService;
    }

    public URL createMetricUrl(String metricName, String metricQuantile) {
        StringBuilder metricPath = new StringBuilder(monitoringUrlPath);
        metricPath.append(metricName);

        if (metricQuantile != null) {
            metricPath.append(".").append(Metric.escapeName(metricQuantile));
        }

        try {
            return new URL("https", graphiteClient.getHost(), metricPath.toString());
        } catch (MalformedURLException e) {
            throw new RuntimeException(e);
        }
    }
}
