package ru.yandex.market.clickphite.monitoring.kronos;

import ru.yandex.market.clickphite.DateTimeUtils;
import ru.yandex.market.clickphite.TimeRange;
import ru.yandex.market.clickphite.graphite.Metric;
import ru.yandex.market.clickphite.monitoring.DataPoint;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 23/11/15
 */
public class KronosModelWrapper {

    private final KronosModel model;
    private final TimeRange timeRange;
    private final int intervalSeconds;

    public KronosModelWrapper(KronosModel model, TimeRange timeRange, int intervalSeconds) {
        this.model = model;
        this.timeRange = timeRange;
        this.intervalSeconds = intervalSeconds;
    }

    public KronosModelWrapper(KronosModel model, LocalDate localDate, int intervalSeconds) {
        this(model, DateTimeUtils.toTimeRange(localDate), intervalSeconds);
    }

    public DataPoint.Status getStatus(int timestampSeconds, double value, boolean checkBottom, boolean checkTop) {
        if (!timeRange.contains(timestampSeconds)) {
            throw new IllegalArgumentException("Timestamp:" + timestampSeconds + " not in " + timeRange);
        }
        int index = (timestampSeconds - timeRange.getStartTimestampSeconds()) / intervalSeconds;
        if ((checkBottom && value < model.getExtBottom(index)) ||
            (checkTop && value > model.getExtTop(index))) {
            return DataPoint.Status.CRITICAL;
        }
        if ((checkBottom && value < model.getBottom(index)) ||
            (checkTop && value > model.getTop(index))) {
            return DataPoint.Status.WARN;
        }
        return DataPoint.Status.OK;
    }

    public TimeRange getTimeRange() {
        return timeRange;
    }

    public List<Metric> getMetrics(String prefix) {
        List<Metric> metrics = new ArrayList<>();
        for (int i = 0; i < model.getSize(); i++) {
            int timestampSeconds = timeRange.getStartTimestampSeconds() + i * intervalSeconds;
            metrics.add(new Metric(timestampSeconds, model.getExtTop(i), prefix + "extTop"));
            metrics.add(new Metric(timestampSeconds, model.getTop(i), prefix + "top"));
            metrics.add(new Metric(timestampSeconds, model.getAverage(i), prefix + "average"));
            metrics.add(new Metric(timestampSeconds, model.getMean(i), prefix + "mean"));
            metrics.add(new Metric(timestampSeconds, model.getBottom(i), prefix + "bottom"));
            metrics.add(
                new Metric(timestampSeconds, model.getExtBottom(i), prefix + "extBottom")
            );
        }
        return metrics;
    }
}
