package ru.yandex.market.clickphite.monitoring.kronos;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.calendar.YandexCalendarDay;
import ru.yandex.market.clickphite.DateTimeUtils;
import ru.yandex.market.clickphite.QueryBuilder;
import ru.yandex.market.clickphite.TimeRange;
import ru.yandex.market.clickphite.config.metric.GraphiteMetricConfig;
import ru.yandex.market.clickphite.config.metric.MetricPeriod;
import ru.yandex.market.clickphite.config.monitoring.KronosConfig;
import ru.yandex.market.clickphite.config.monitoring.MonitoringConfig;
import ru.yandex.market.clickphite.graphite.Metric;
import ru.yandex.market.clickphite.monitoring.DataPoint;
import ru.yandex.market.clickphite.monitoring.MonitoringContext;
import ru.yandex.market.clickphite.monitoring.MonitoringService;
import ru.yandex.market.clickphite.monitoring.MonitoringType;

import javax.net.ssl.SSLHandshakeException;
import java.io.IOException;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 05/11/15
 */
public class KronosMonitoringContext extends MonitoringContext {

    private static final Logger log = LogManager.getLogger();

    private static final MetricPeriod MODEL_PERIOD = MetricPeriod.FIVE_MIN;
    private static final int POINTS_IN_MODEL = (int) TimeUnit.DAYS.toSeconds(1) / MODEL_PERIOD.getDurationSeconds();

    private final KronosConfig kronosConfig;


    private final Cache<LocalDate, KronosModelWrapper> models = CacheBuilder.newBuilder()
        .expireAfterWrite(1, TimeUnit.DAYS)
        .build();


    public KronosMonitoringContext(GraphiteMetricConfig metricConfig, MonitoringConfig monitoringConfig,
                                   MonitoringService monitoringService, String fieldName) {
        super(metricConfig, monitoringConfig, monitoringService, fieldName);
        kronosConfig = monitoringConfig.getKronos();
    }

    @Override
    public MonitoringType getType() {
        return MonitoringType.KRONOS;
    }

    @Override
    public void onDataPoint(DataPoint dataPoint) {
        LocalDate date = DateTimeUtils.toLocalDate(dataPoint.getTimestampSeconds());

        KronosModelWrapper model = models.getIfPresent(date);
        if (model == null) {
            try {
                model = loadModel(date, monitoringService, metricConfig);
            } catch (SSLHandshakeException e) {
                log.warn("SSL handshake error: " + e.getMessage());
                return;
            } catch (Exception e) {
                throw new RuntimeException("Error in creating kronos model", e);
            }
        }

        DataPoint.Status status = model.getStatus(
            dataPoint.getTimestampSeconds(), dataPoint.getValue(),
            kronosConfig.isCheckBottom(), kronosConfig.isCheckTop()
        );
        dataPoint.setStatus(status);
    }

    private KronosModelWrapper loadModel(LocalDate date, MonitoringService monitoringService,
                                         GraphiteMetricConfig metricConfig) throws IOException, InterruptedException {

        log.info("Loading kronos model for metric: " + metricConfig.toString() + ", date: " + date);
        List<YandexCalendarDay> days = monitoringService.getSimilarDays(date, kronosConfig.getSimilarDaysCount());

        List<double[]> trainingData = getTrainingData(days, metricConfig, monitoringService);
        KronosModel kronosModel = KronosTrainer.estimateConfidenceIntervals(
            trainingData, kronosConfig.getSmoothness(), kronosConfig.getMult()
        );
        KronosModelWrapper modelWrapper = new KronosModelWrapper(kronosModel, date, MODEL_PERIOD.getDurationSeconds());
        models.put(date, modelWrapper);
        String graphitePrefix = getGraphitePrefix(metricConfig);
        log.info("Sending kronos data to graphite with prefix: " + graphitePrefix);
        List<Metric> metrics = modelWrapper.getMetrics(graphitePrefix);
        monitoringService.getGraphiteClient().send(metrics);
        log.info("Done loading kronos model for metric: " + metricConfig.toString() + ", date: " + date);
        return modelWrapper;
    }

    private static String getGraphitePrefix(GraphiteMetricConfig metricConfig) {
        return metricConfig.getPeriod().getGraphiteName() + "." + metricConfig.getMetricPrefix() +
            "kronos." + metricConfig.getMetricName() + ".";
    }

    private List<double[]> getTrainingData(List<YandexCalendarDay> days,
                                           GraphiteMetricConfig metricConfig, MonitoringService monitoringService) {

        if (metricConfig.getPeriod() != MetricPeriod.FIVE_MIN) {
            metricConfig = metricConfig.copy();
            metricConfig.setPeriod(MODEL_PERIOD);
        }
        String queryTemplate = QueryBuilder.buildMetricQueryTemplate(metricConfig);

        List<double[]> trainData = new ArrayList<>(days.size());

        for (YandexCalendarDay day : days) {
            double[] points = new double[POINTS_IN_MODEL];
            trainData.add(points);
            Arrays.fill(points, Double.NaN);
            TimeRange timeRange = DateTimeUtils.toTimeRange(day.getDate());
            String query = QueryBuilder.placeTimeConditionToQuery(queryTemplate, timeRange, 1);
            monitoringService.getClickhouseTemplate().query(
                query,
                rs -> {
                    int relativeTimestampSeconds =
                        rs.getInt(QueryBuilder.TIMESTAMP_INDEX) - timeRange.getStartTimestampSeconds();

                    int pointIndex = relativeTimestampSeconds / MODEL_PERIOD.getDurationSeconds();
                    points[pointIndex] = rs.getDouble(QueryBuilder.VALUE_INDEX);
                }
            );
        }
        return trainData;
    }
}
