package ru.yandex.market.clickphite.monitoring.range;

import ru.yandex.market.clickphite.DateTimeUtils;
import ru.yandex.market.clickphite.config.metric.GraphiteMetricConfig;
import ru.yandex.market.clickphite.config.monitoring.MonitoringConfig;
import ru.yandex.market.clickphite.config.monitoring.RangeConfig;
import ru.yandex.market.clickphite.monitoring.DataPoint;
import ru.yandex.market.clickphite.monitoring.MonitoringContext;
import ru.yandex.market.clickphite.monitoring.MonitoringService;
import ru.yandex.market.clickphite.monitoring.MonitoringType;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 22/01/16
 */
public class RangeMonitoringContext extends MonitoringContext {
    private static final int MAX_FRACTION_DIGITS = 4;

    private static final ThreadLocal<SimpleDateFormat> DATE_FORMAT =
        ThreadLocal.withInitial(() -> new SimpleDateFormat("HH:mm:ss"));

    private static final ThreadLocal<NumberFormat> VALUE_FORMAT =
        ThreadLocal.withInitial(RangeMonitoringContext::createNumberFormat);

    private RangeConfig rangeConfig;


    public RangeMonitoringContext(GraphiteMetricConfig metricConfig, MonitoringConfig monitoringConfig,
                                  MonitoringService monitoringService, String fieldName) {
        super(metricConfig, monitoringConfig, monitoringService, fieldName);
        this.rangeConfig = monitoringConfig.getRange();
    }

    private static DecimalFormat createNumberFormat() {
        DecimalFormatSymbols decimalFormatSymbols = DecimalFormatSymbols.getInstance(Locale.US);

        DecimalFormat decimalFormat = new DecimalFormat();
        decimalFormat.setMaximumFractionDigits(MAX_FRACTION_DIGITS);
        decimalFormat.setDecimalFormatSymbols(decimalFormatSymbols);
        decimalFormat.setGroupingUsed(true);
        return decimalFormat;
    }

    @Override
    public MonitoringType getType() {
        return MonitoringType.RANGE;
    }

    @Override
    public void onDataPoint(DataPoint dataPoint) {
        DataPoint.Status status = getStatus(dataPoint.getValue());
        dataPoint.setStatus(status);
    }

    private DataPoint.Status getStatus(double value) {
        if (Double.isNaN(value)) {
            return DataPoint.Status.UNKNOWN;
        }
        if (value >= rangeConfig.getCritTop() || value <= rangeConfig.getCritBottom()) {
            return DataPoint.Status.CRITICAL;
        }
        if (value >= rangeConfig.getWarnTop() || value <= rangeConfig.getWarnBottom()) {
            return DataPoint.Status.WARN;
        }
        return DataPoint.Status.OK;
    }

    @Override
    public String getMetricString() {
        Stream.Builder<String> outerBuilder = Stream.builder();

        buildRanges(rangeConfig.getWarnBottom(), rangeConfig.getWarnTop(), "WARN")
            .ifPresent(outerBuilder::accept);

        buildRanges(rangeConfig.getCritBottom(), rangeConfig.getCritTop(), "CRIT")
            .ifPresent(outerBuilder::accept);

        return outerBuilder.build().collect(Collectors.joining("; ", "", "\n"));
    }

    private Optional<String> buildRanges(double bottom, double top, String title) {
        Optional<String> optionalBottom = isSpecified(bottom)
            .map(it -> "value >= " + it);
        Optional<String> optionalTop = isSpecified(top)
            .map(it -> "value <= " + it);

        List<String> values = Stream.of(optionalBottom, optionalTop)
            .filter(Optional::isPresent)
            .map(Optional::get)
            .collect(Collectors.toList());

        if (values.isEmpty()) {
            return Optional.empty();
        }

        return Optional.of(title + ": " + values.stream().collect(Collectors.joining(", ")));
    }

    @Override
    public String getCauseString() {
        return currentStatus.getDataPoints().stream()
            .sorted(Comparator.comparing(DataPoint::getTimestampSeconds))
            .map(dp -> "{ " + formatDate(dp.getTimestampSeconds()) + ": " + formatValue(dp.getValue()) + " }")
            .collect(Collectors.joining(", ", "[", "]"));
    }

    private static String formatDate(int seconds) {
        return DATE_FORMAT.get().format(DateTimeUtils.dateFromTimeStampSeconds(seconds));
    }

    private static String formatValue(double value) {
        return VALUE_FORMAT.get().format(value);
    }

    private static Optional<Double> isSpecified(double value) {
        return Optional.of(value)
            .filter(d -> !d.isNaN());
    }
}
