package ru.yandex.market.clickphite.solomon;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.hash.Hashing;
import com.google.gson.Gson;
import org.apache.commons.io.IOUtils;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.clickphite.solomon.dto.SolomonCluster;
import ru.yandex.market.clickphite.solomon.dto.SolomonPushRequestBody;
import ru.yandex.market.clickphite.solomon.dto.SolomonPushResponseBody;
import ru.yandex.market.clickphite.solomon.dto.SolomonSensor;
import ru.yandex.market.clickphite.solomon.dto.SolomonService;
import ru.yandex.market.clickphite.solomon.dto.SolomonShard;
import ru.yandex.market.health.KeyValueLog;
import ru.yandex.market.request.trace.TskvRecordBuilder;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * https://wiki.yandex-team.ru/solomon/api/push/.
 *
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 18.07.2018
 */
public class SolomonClient {
    private static final Logger log = LogManager.getLogger();
    private static final Logger solomonClientLog = LogManager.getLogger("solomon_client");
    private static final Logger solomonClientShardsLog = LogManager.getLogger("solomon_client_shards");
    private static final Logger solomonClientRequestBodiesLog = LogManager.getLogger("solomon_client_request_bodies");

    // Просто ContentType.APPLICATION_JSON не катит, потому что Соломон не понимает Content-Type с кодировкой.
    private static final ContentType PUSH_CONTENT_TYPE = ContentType.create(ContentType.APPLICATION_JSON.getMimeType());
    public static final int SOLOMON_SHARD_ID_LENGTH_LIMIT = 55;

    private final CloseableHttpClient httpClient;
    private final Gson gson = new Gson();

    private final String baseUrl;
    private final String token;
    private final boolean fullAcknowledge;
    private final boolean writeRequestLog;

    public SolomonClient(String baseUrl, String token, int timeoutSeconds,
                         boolean fullAcknowledge, boolean writeRequestLog) {
        this.baseUrl = baseUrl;
        this.token = token;
        this.fullAcknowledge = fullAcknowledge;
        this.writeRequestLog = writeRequestLog;
        httpClient = HttpClientBuilder.create()
            .setDefaultRequestConfig(
                RequestConfig.custom()
                    .setConnectTimeout((int) TimeUnit.SECONDS.toMillis(timeoutSeconds))
                    .setConnectionRequestTimeout((int) TimeUnit.SECONDS.toMillis(timeoutSeconds))
                    .setSocketTimeout((int) TimeUnit.SECONDS.toMillis(timeoutSeconds))
                    .build()
            )
            .setMaxConnTotal(100)
            .setMaxConnPerRoute(100)
            .build();

    }


    public SolomonClient(String baseUrl, String token, int timeoutSeconds) {
        this(baseUrl, token, timeoutSeconds, true, false);
    }


    /**
     * @return Количество сохранённых сенсоров.
     */
    public long push(String contextIdForLog, SolomonShardId shardId, SolomonPushRequestBody sensors) {
        solomonClientRequestBodiesLog.info(
            "Pushing {} points. Shard={}, id={}",
            sensors.getSensors().size(), shardId, contextIdForLog
        );

        try {
            URI uri = buildPushUri(shardId);
            String requestBody = gson.toJson(sensors);

            writeToSolomonClientLog(contextIdForLog, shardId, sensors, requestBody);
            if (writeRequestLog) {
                writeToSolomonClientRequestsLog(uri, requestBody);
            }

            try (CloseableHttpResponse response = httpClient.execute(buildPushRequest(uri, requestBody))) {
                return checkPushResponseAndReturnSavedSensorsCount(response);
            }
        } catch (IOException | URISyntaxException e) {
            throw new SolomonClientException("Failed to push sensors to Solomon for shard " + shardId, e);
        }
    }

    public void createServiceAndClusterAndShard(SolomonShardId shardId) {
        createService(shardId.getProject(), shardId.getService());
        createCluster(shardId.getProject(), shardId.getCluster());
        createShard(shardId.getProject(), shardId.getCluster(), shardId.getService());
    }

    private void createService(String project, String service) {
        createEntity(
            "service",
            String.format("api/v2/projects/%s/services", project),
            new SolomonService(
                project + "_" + service,
                service,
                SolomonService.Type.JSON_GENERIC
            )
        );
    }

    private void createCluster(String project, String cluster) {
        createEntity(
            "cluster",
            String.format("api/v2/projects/%s/clusters", project),
            new SolomonCluster(
                project + "_" + cluster,
                cluster
            )
        );
    }

    private void createShard(String project, String cluster, String service) {
        createEntity(
            "shard",
            String.format("api/v2/projects/%s/shards", project),
            new SolomonShard(
                getShardId(project, cluster, service),
                project,
                project + "_" + service,
                project + "_" + cluster
            )
        );
    }

    @VisibleForTesting
    String getShardId(String project, String cluster, String service) {
        String id = project + "_" + cluster + "_" + service;

        // id must have length fewer than 55 characters"
        // but we keep the same id generation for already posted id to solomon
        if (id.length() >= SOLOMON_SHARD_ID_LENGTH_LIMIT) {
            id = Hashing.murmur3_128().hashString(id, StandardCharsets.UTF_8).toString();
        }
        return id;
    }

    private void createEntity(String entityNameForLog, String path, Object entity) {
        solomonClientShardsLog.info(KeyValueLog.format(new Date(), "create_attempt", entityNameForLog, 1));
        try {
            HttpUriRequest request = RequestBuilder.post(baseUrl + path)
                .addHeader("Authorization", "OAuth " + token)
                .addHeader("Accept", ContentType.APPLICATION_JSON.getMimeType())
                .setEntity(new StringEntity(gson.toJson(entity), PUSH_CONTENT_TYPE))
                .build();

            //remove token for logs
            HttpUriRequest printableRequest = RequestBuilder.copy(request)
                .setHeader("Authorization", "OAuth ****-****-****-****")
                .build();

            log.info(printableRequest);
            for (Header header : printableRequest.getAllHeaders()) {
                log.info("{}: {}", header.getName(), header.getValue());
            }
            log.info(gson.toJson(entity));

            try (CloseableHttpResponse response = httpClient.execute(request)) {
                int statusCode = response.getStatusLine().getStatusCode();
                String responseBody = IOUtils.toString(response.getEntity().getContent(), StandardCharsets.UTF_8);

                log.info(responseBody);

                if (statusCode == HttpStatus.SC_CONFLICT) {
                    solomonClientShardsLog.info(KeyValueLog.format(new Date(), "already_exists", entityNameForLog, 1));
                    return;
                }

                if (is2xx(statusCode)) {
                    solomonClientShardsLog.info(KeyValueLog.format(new Date(), "created", entityNameForLog, 1));
                    return;
                }

                solomonClientShardsLog.info(KeyValueLog.format(new Date(), "create_failed", entityNameForLog, 1));
                throw new SolomonClientException(String.format(
                    "Failed to create %s %s: %s",
                    entityNameForLog, path, responseBody
                ));
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private URI buildPushUri(SolomonShardId shardId) throws URISyntaxException {
        return new URIBuilder(baseUrl)
            .setPath("/push")
            .setParameter("project", shardId.getProject())
            .setParameter("service", shardId.getService())
            .setParameter("cluster", shardId.getCluster())
            .setParameter("ack", fullAcknowledge ? "full" : "at_least_one")
            .build();
    }

    private static HttpUriRequest buildPushRequest(URI uri, String requestBody) {
        return RequestBuilder.create(HttpPost.METHOD_NAME)
            .setUri(uri)
            .setEntity(buildPushRequestEntity(requestBody))
            .build();
    }

    // MARKETINFRA-4744 Без указания кодировки используется кодировка по умолчанию Consts.ISO_8859_1
    // Соломон не понимает Content-Type с кодировкой, приходится указывать ContentType c UTF-8 а потом переопределять
    @VisibleForTesting
    static StringEntity buildPushRequestEntity(String requestBody) {
        StringEntity entity = new StringEntity(requestBody, ContentType.APPLICATION_JSON);
        entity.setContentType(PUSH_CONTENT_TYPE.toString());
        return entity;
    }

    private static void writeToSolomonClientLog(String contextId, SolomonShardId shardId,
                                                SolomonPushRequestBody sensors, String requestBody) {
        TskvRecordBuilder builder = new TskvRecordBuilder();
        builder.add("date", Instant.now());
        builder.add("context_id", contextId);
        builder.add("project", shardId.getProject());
        builder.add("service", shardId.getService());
        builder.add("cluster", shardId.getCluster());
        builder.add("request_body_size_bytes", requestBody.length());
        builder.add("sensors_count", sensors.getSensors().size());
        builder.add("common_labels_count", sensors.getCommonLabels().size());
        builder.add(
            "average_not_common_labels_count",
            sensors.getSensors().stream()
                .map(SolomonSensor::getLabels)
                .mapToInt(Map::size)
                .average()
                .orElse(0)
        );
        builder.add(
            "unique_not_common_label_sets_count",
            sensors.getSensors().stream()
                .map(SolomonSensor::getLabels)
                .distinct()
                .count()
        );
        solomonClientLog.info(builder.build());
    }

    private static void writeToSolomonClientRequestsLog(URI uri, String requestBody) {
        solomonClientRequestBodiesLog.info("Solomon request: url={} body={}", uri, requestBody);
    }

    private static long checkPushResponseAndReturnSavedSensorsCount(HttpResponse response) throws IOException {
        return checkPushResponseAndReturnSavedSensorsCount(
            response.getStatusLine().getStatusCode(),
            EntityUtils.toString(response.getEntity())
        );
    }

    /**
     * Этот метод проверяет что Соломон сказал что всё ок и все сенсоры сохранились.
     * <p>
     * Нельзя просто проверить код ответа, потому что если какие-то сенсоры в теле запроса невалидны, то Соломон их
     * просто пропустит, вернёт 200 и в теле ответа напишет что запушилось не всё. Пример: на попытку запушить сенсоры с
     * невалидными лейблами вернулось 200 и нули в теле: https://paste.yandex-team.ru/512792.
     * <p>
     * Пример того, что возвращает Соломон в теле ответа: https://paste.yandex-team.ru/521831. Это парсится в
     * {@link SolomonPushResponseBody#parse(String)}.
     * <p>
     * В Аркадии можно поискать строку "successful sensors" и найти код Соломона. Там сенсоры пушатся на реплики
     * Соломона, каждая реплика возвращает количество сенсоров, которые успешно запушились, и ответы реплик
     * агрегируются.
     * <p>
     * Число до тильды - это количество сенсоров, которые запушились на реплику, на которую запушилось МЕНЬШЕ всего
     * сенсоров.
     * Число после тильды - это количество сенсоров, которые запушились на реплику, на которую запушилось БОЛЬШЕ всего
     * сенсоров.
     * <p>
     * У ручки /push есть ещё незадокументированный параметр ack, который по умолчанию full, это означает что ручка
     * должна возвращать успех только если сенсоры запушились на все шарды. То есть когда всё хорошо, до и после тильды
     * должно быть одно и то же число, количество метрик, которые мы отправили в теле запроса.
     * <p>
     * Если в ответе Соломона оба числа меньше чем количество сенсоров, которое мы отправили, значит часть сенсоров не
     * прошла валидацию. MARKETINFRA-3906. Это ок, количество таким образом проигнорированных сенсоров запишется в
     * metric.log в поле invalid_rows_ignored_per_id.
     */
    @VisibleForTesting
    static long checkPushResponseAndReturnSavedSensorsCount(int responseCode, String responseBody) {
        solomonClientRequestBodiesLog.info(
            "Sent sensors to Solomon. {}",
            responseCodeAndBodyToString(responseCode, responseBody)
        );

        if (responseCode == HttpStatus.SC_NOT_FOUND) {
            throw new SolomonShardNotFoundException();
        }

        if (!is2xx(responseCode)) {
            throw new SolomonClientException(
                "Failed to send sensors to Solomon. " + responseCodeAndBodyToString(responseCode, responseBody)
            );
        }

        SolomonPushResponseBody parsedResponseBody = SolomonPushResponseBody.parse(responseBody);

        if (parsedResponseBody.getMinSuccessfulSensorCount() != parsedResponseBody.getMaxSuccessfulSensorCount()) {
            // Соломон не смог сохранить часть сенсоров. Бросаем исключение чтобы поретраить.
            throw new SolomonClientException(String.format(
                "Solomon didn't save all sensors. %s",
                responseCodeAndBodyToString(responseCode, responseBody)
            ));
        }

        return parsedResponseBody.getMinSuccessfulSensorCount();
    }

    @SuppressWarnings("MagicNumber")
    private static boolean is2xx(int statusCode) {
        return statusCode >= 200 && statusCode < 300;
    }

    private static String responseCodeAndBodyToString(int responseCode, String responseBody) {
        return String.format("Response: code=%s, body: '\n%s\n'", responseCode, responseBody);
    }
}
