package ru.yandex.market.clickphite.solomon.dto;

import com.google.gson.Gson;
import ru.yandex.market.clickphite.solomon.SolomonClientException;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * В ответе Соломона не JSON, а просто текст, поэтому парсим не GSON'ом, а методом {@link #parse}.
 * <p>
 * Пример того, что возвращает Соломон: https://paste.yandex-team.ru/521831
 *
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 08.08.2018
 */
public class SolomonPushResponseBody {
    private static final Gson GSON = new Gson();

    private static final Pattern SUCCESSFUL_AND_TOTAL_SENSORS_PATTERN =
        Pattern.compile("^successful sensors (\\d+)~(\\d+);", Pattern.MULTILINE);

    private final long minSuccessfulSensorCount;
    private final long maxSuccessfulSensorCount;

    private SolomonPushResponseBody(long minSuccessfulSensorCount, long maxSuccessfulSensorCount) {
        this.minSuccessfulSensorCount = minSuccessfulSensorCount;
        this.maxSuccessfulSensorCount = maxSuccessfulSensorCount;
    }

    public long getMinSuccessfulSensorCount() {
        return minSuccessfulSensorCount;
    }

    public long getMaxSuccessfulSensorCount() {
        return maxSuccessfulSensorCount;
    }

    public static SolomonPushResponseBody parse(String s) {
        try {
            Response response = GSON.fromJson(s, Response.class);
            if (!response.status.equals("OK")) {
                throw new SolomonClientException(
                    "Status is not OK, status: " + response.status
                );
            }
            return new SolomonPushResponseBody(
                response.sensorsProcessed, response.sensorsProcessed
            );
        } catch (RuntimeException ignored) {
            // goto fallback
        }

        // todo remove fallback when api become persistent
        Matcher matcher = SUCCESSFUL_AND_TOTAL_SENSORS_PATTERN.matcher(s);
        if (!matcher.find()) {
            throw new SolomonClientException(
                String.format("Can't find successful and total sensors in string: '\n%s\n'", s)
            );
        }
        try {
            return new SolomonPushResponseBody(
                Long.parseLong(matcher.group(1)),
                Long.parseLong(matcher.group(2))
            );
        } catch (NumberFormatException e) {
            throw new SolomonClientException(
                String.format("Can't parse successful and total sensors in string: '\n%s\n'", s), e
            );
        }
    }

    private static class Response {
        String status;
        long sensorsProcessed;
    }
}
