package ru.yandex.market.clickphite.solomon.dto;

import com.google.common.collect.Maps;
import com.google.gson.annotations.SerializedName;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * https://wiki.yandex-team.ru/solomon/api/push/.
 * https://wiki.yandex-team.ru/solomon/api/dataformat/json/.
 *
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 18.07.2018
 */
public class SolomonSensor {
    private final Map<String, CharSequence> labels;
    private final Kind kind;
    private final Number value;  // В зависимости от kind тут может быть целое или дробное число
    @SerializedName("ts")
    private final Long timestamp;
    private final List<TimestampValuePair> timeseries;  // Должны быть отсортированы по timestamp'у!
    private final Boolean memOnly;

    /**
     * @param labels CharSequence вместо String чтобы можно было передавать StringBuilder не вызывая у него toString.
     */
    public SolomonSensor(Map<String, CharSequence> labels, Kind kind, Number value, Long timestamp,
                         List<TimestampValuePair> timeseries, Boolean memOnly) {
        this.labels = labels;
        this.kind = kind;
        this.value = value;
        this.timestamp = timestamp;
        this.timeseries = timeseries == null ? null : sortByTimestamp(timeseries);
        this.memOnly = memOnly;
    }

    private static List<TimestampValuePair> sortByTimestamp(List<TimestampValuePair> timeseries) {
        return timeseries.stream()
            .sorted(Comparator.comparingLong(pair -> pair.timestamp))
            .collect(Collectors.toList());
    }

    public Map<String, String> getLabels() {
        return Maps.transformValues(labels, CharSequence::toString);
    }

    public Kind getKind() {
        return kind;
    }

    public Number getValue() {
        return value;
    }

    public Long getTimestamp() {
        return timestamp;
    }

    public List<TimestampValuePair> getTimeseries() {
        return timeseries;
    }

    public Boolean getMemOnly() {
        return memOnly;
    }

    /**
     * Подробнее о том, что все эти типы значат на https://wiki.yandex-team.ru/solomon/api/dataformat/json/.
     */
    public enum Kind {
        DGAUGE,
        IGAUGE,
        COUNTER,
        RATE,
        HIST,
        HIST_RATE,
    }

    public static class TimestampValuePair {
        @SerializedName("ts")
        private final long timestamp;
        private final Number value;  // В зависимости от kind тут может быть целое или дробное число

        public TimestampValuePair(long timestamp, Number value) {
            this.timestamp = timestamp;
            this.value = value;
        }

        public long getTimestamp() {
            return timestamp;
        }

        public Number getValue() {
            return value;
        }
    }
}
