package ru.yandex.market.clickphite.worker;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Range;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.clickphite.ClickphiteService;
import ru.yandex.market.clickphite.DateTimeUtils;
import ru.yandex.market.clickphite.TimeRange;
import ru.yandex.market.clickphite.metric.AsyncMetricGroupMonitoring;
import ru.yandex.market.clickphite.metric.MetricContext;
import ru.yandex.market.clickphite.metric.MetricContextGroup;
import ru.yandex.market.clickphite.metric.MetricContextGroupImpl;
import ru.yandex.market.clickphite.metric.MetricQueue;
import ru.yandex.market.clickphite.metric.MetricService;
import ru.yandex.market.clickphite.metric.QueryWeight;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author Anton Sukhonosenko <a href="mailto:algebraic@yandex-team.ru"></a>
 * @date 20.09.17
 */
public class MetricGroupWorker extends Worker {
    private static final Logger log = LogManager.getLogger();

    private final MetricService metricService;
    private final AsyncMetricGroupMonitoring asyncMetricGroupMonitoring;
    private final MetricContextGroup metricGroup;

    private final Map<MetricContext, MetricQueue> queueMap;
    private final QueryWeight queryWeight;

    private Supplier<Long> currentTimeMillis = System::currentTimeMillis;

    public MetricGroupWorker(ClickphiteService clickphiteService, MetricService metricService,
                             AsyncMetricGroupMonitoring asyncMetricGroupMonitoring, MetricContextGroup metricGroup,
                             QueryWeight queryWeight) {
        super(
            clickphiteService, metricGroup.getProcessStatus(),
            "Metric group updater: " + metricGroup.toString()
        );
        this.metricService = metricService;
        this.asyncMetricGroupMonitoring = asyncMetricGroupMonitoring;
        this.metricGroup = metricGroup;
        this.queryWeight = queryWeight;
        this.queueMap = metricGroup.getMetricContexts().stream()
            .collect(Collectors.toMap(Function.identity(), MetricContext::getMetricQueue));
    }

    /**
     * В первую очередь строим свежие метрики, назад ограничиваем себя maxQueriesPerUpdate
     * Удаляем из очереди только законченные задачи.
     *
     * @throws Exception
     */
    @Override
    public void doWork() throws Exception {
        ClickphiteService clickphiteService = getClickphiteService();
        int metricDelaySeconds = clickphiteService.getMetricDelaySeconds();
        int currentTimeSeconds = (int) TimeUnit.MILLISECONDS.toSeconds(currentTimeMillis.get());
        int metricEndTimeSeconds = metricGroup.getPeriod().getEndTimeSeconds(metricDelaySeconds, currentTimeSeconds);
        long maxCompactionTimeMillis = currentTimeMillis.get() - TimeUnit.SECONDS.toMillis(metricDelaySeconds);

        Stream<MetricRange> metricRanges = queueMap.entrySet()
            .stream()
            .flatMap(e -> {
                MetricContext metricContext = e.getKey();
                MetricQueue queue = e.getValue();
                List<Range<Integer>> ranges = queue.compactAndGet(metricEndTimeSeconds, maxCompactionTimeMillis);
                return ranges.stream().map(r -> new MetricRange(metricContext, r));
            });

        Stream<MetricTimeRange> metricTimeRanges = metricRanges
            // TODO: можно оптимизировать
            // https://stackoverflow.com/questions/31575043/performance-of-stream-sorted-limit
            .sorted(MetricRange.DESCENDING_COMPARATOR)
            .flatMap(metricRange -> {
                List<TimeRange> timeRanges = DateTimeUtils.sliceToTimeRanges(
                    metricRange.range, metricGroup.getPeriod(), clickphiteService.getMaxQueriesPerUpdate(), false
                );
                return timeRanges.stream().map(r -> new MetricTimeRange(metricRange.metric, r));
            });

        List<TimeRangeGroup> timeRangeGroups = metricTimeRanges.collect(
            Collectors.groupingBy(mr -> mr.timeRange, Collectors.mapping(mr -> mr.metric, Collectors.toList()))
        )
            .entrySet()
            .stream()
            .map(e -> new TimeRangeGroup(e.getKey(), e.getValue()))
            .sorted(TimeRangeGroup.DESCENDING_COMPARATOR)
            .limit(clickphiteService.getMaxQueriesPerUpdate())
            .collect(Collectors.toList());

        for (TimeRangeGroup timeRangeGroup : timeRangeGroups) {
            List<MetricContext> metricContexts = timeRangeGroup.metricContexts;
            TimeRange timeRange = timeRangeGroup.timeRange;

            MetricContextGroup metricGroupToUpdate = metricGroup;
            if (metricContexts.size() < metricGroup.getMetricContexts().size()) {
                metricGroupToUpdate = MetricContextGroupImpl.create(metricContexts, metricGroup);
            }

            if (!shouldBeBuiltNow(timeRange)) {
                return;
            }

            log.debug("Updating for time range {}, group {}", timeRange, metricGroupToUpdate.toString());
            long updateTimeMillis = currentTimeMillis.get() - TimeUnit.SECONDS.toMillis(metricDelaySeconds);
            metricService.updateMetricGroup(metricGroupToUpdate, timeRange, getQueryWeight());

            for (MetricContext metricContext : metricGroupToUpdate.getMetricContexts()) {
                MetricQueue queue = this.queueMap.get(metricContext);
                queue.remove(updateTimeMillis, timeRange.getRange());
            }
        }

        log.debug("Finishing metric group worker");
    }

    private boolean shouldBeBuiltNow(TimeRange timeRange) {
        if (!getClickphiteService().isLagMode()) {
            return true;
        }
        int maxTimestampSeconds = (int) TimeUnit.MILLISECONDS.toSeconds(currentTimeMillis.get()) -
            (int) TimeUnit.MINUTES.toSeconds(getClickphiteService().getMaxMetricsAgeOnLagMinutes());
        return timeRange.getEndTimestampSeconds() + metricGroup.getPeriod().getDurationSeconds() >= maxTimestampSeconds;
    }

    @Override
    public QueryWeight getQueryWeight() {
        return queryWeight;
    }

    @Override
    protected void statusUpdated() {
        if (getStatus().isError()) {
            asyncMetricGroupMonitoring.reportFailure(metricGroup);
        } else {
            asyncMetricGroupMonitoring.reportSuccess(metricGroup);
        }
    }

    @VisibleForTesting
    void setCurrentTimeMillisSupplier(Supplier<Long> currentTimeMillis) {
        this.currentTimeMillis = currentTimeMillis;
    }

    static class MetricRange {
        private static final Comparator<MetricRange> DESCENDING_COMPARATOR = Comparator
            .<MetricRange>comparingInt(mr -> -mr.range.upperEndpoint())
            .thenComparingInt(mr -> mr.range.lowerEndpoint());

        private final MetricContext metric;
        private final Range<Integer> range;

        MetricRange(MetricContext metric, Range<Integer> range) {
            this.metric = metric;
            this.range = range;
        }
    }

    static class MetricTimeRange {
        private final MetricContext metric;
        private final TimeRange timeRange;

        MetricTimeRange(MetricContext metric, TimeRange timeRange) {
            this.metric = metric;
            this.timeRange = timeRange;
        }
    }

    static class TimeRangeGroup {
        private static final Comparator<TimeRangeGroup> DESCENDING_COMPARATOR = Comparator
            .<TimeRangeGroup>comparingInt(g -> -g.timeRange.getEndTimestampSeconds())
            .thenComparingInt(g -> g.timeRange.getStartTimestampSeconds());

        private final TimeRange timeRange;
        private final List<MetricContext> metricContexts;

        TimeRangeGroup(TimeRange timeRange, List<MetricContext> metricContexts) {
            this.timeRange = timeRange;
            this.metricContexts = metricContexts;
        }

    }
}
