#!/usr/bin/python

import argparse
import json
import os

import time
import datetime
import dateutil.parser

CONFIGS_PATH = os.path.sep.join(['src', 'configs', 'test'])

parser = argparse.ArgumentParser()
parser.add_argument('--file-name', required=True, type=str,
                    help='File name from directory %s' % CONFIGS_PATH)

PROJECT_PATH = os.path.dirname(os.path.realpath(__file__))

while not os.path.exists(os.path.join(PROJECT_PATH, CONFIGS_PATH)):
    PROJECT_PATH = os.path.dirname(PROJECT_PATH)

def detect_value_type(value):
    # type: (str, str) -> dict
    comma_index = value.find(',')
    if comma_index > 0:
        first_item = value[0:comma_index]
        _, type = detect_value_type(first_item)
        return value, 'Array(%s)' % type

    try:
        converted_value = int(value)
        return converted_value, 'Int32'
    except ValueError:
        pass

    try:
        converted_value = dateutil.parser.parse(value)
        return converted_value, 'DateTime'
    except ValueError:
        pass

    return value, 'String'

def process_row(data_row):
    # type: (str) -> dict
    row_split = data_row.split('\t')
    processed_data = {}
    for kv in row_split[1:]:
        key, value = kv.split('=')
        #print value
        converted_value, type = detect_value_type(value)
        processed_data[key] = {'type': type, 'key': key, 'value': value, 'converted_value': converted_value}
    return processed_data

def read_row(f_path):
    with open(os.path.join(PROJECT_PATH, CONFIGS_PATH, f_path)) as f:
        return f.readline()

if __name__ == '__main__':
    args = parser.parse_args()

    file_name = args.file_name
    row = read_row(file_name)
    columns = process_row(row)
    output_columns = {}
    for key, value in columns.iteritems():
        if key != 'date' and key != 'timestamp':
            output_columns[key] = {'type': value['type']}

    result = {
        'logHosts': '*',
        'logPath': "<logs path>",
        "clickhouseTable": "<table name>",
        'parser': {
            "dateFormat": "yyyy-MM-dd'T'HH:mm:ss.SSSX",
            "tskvMatch": {},
            "separator": "\t",
            "columns": output_columns
        }
    }

    target_name = file_name[0:file_name.find('.')]
    target_file = os.path.join(PROJECT_PATH, 'src', 'configs', target_name + '.json')
    with open(target_file, 'w') as f:
        f.write(json.dumps(result, sort_keys=True, indent=2, separators=(',', ': ')))

    # test generation todo
    expected = {}
    for key, value in columns.iteritems():
        if key != 'date' and key != 'timestamp':
            if value['type'].startswith('Date'):
                expected[key] = value['value']
            else:
                expected[key] = value['converted_value']

    expected['timestampSeconds'] = time.mktime(columns['date']['converted_value'].timetuple()) + 10800

    result = {
        "config": target_name + ".json",
        "input": target_name + ".txt",
        "expected": [expected]
    }

    target_file = os.path.join(PROJECT_PATH, 'src', 'configs', 'test', target_name + '.json')
    with open(target_file, 'w') as f:
        f.write(json.dumps(result, sort_keys=True, indent=2, separators=(',', ': ')))
