package ru.yandex.market.clickhouse.ddl;

import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import java.util.ArrayList;
import java.util.concurrent.TimeUnit;

/**
 * @author Dmitry Poluyanov <a href="https://t.me/neiwick">Dmitry Poluyanov</a>
 * @since 12.07.17
 * <p>
 * Run tests BEFORE any changes in code (or on code from {@code master} branch) and store results.
 * Then write performance tests to your new functionality and re-run tests again (and again, and again).
 * <p>
 * Next table is only for example! Reproduce test on your machine for obtain actual results.
 * <pre>
 * {@code
 *
 * Benchmark                                                  (arraySize)  Mode  Cnt    Score   Error  Units
 * ColumnTypeBenchmark.avgTimeOfStringArrayTypeWithArray                0  avgt    5    5,587 ± 0,230  ns/op
 * ColumnTypeBenchmark.avgTimeOfStringArrayTypeWithArray                2  avgt    5   15,570 ± 1,067  ns/op
 * ColumnTypeBenchmark.avgTimeOfStringArrayTypeWithArray                8  avgt    5   45,054 ± 0,967  ns/op
 * ColumnTypeBenchmark.avgTimeOfStringArrayTypeWithArray               16  avgt    5   83,588 ± 3,938  ns/op
 * ColumnTypeBenchmark.avgTimeOfStringArrayTypeWithArray               32  avgt    5  168,478 ± 5,985  ns/op
 * ColumnTypeBenchmark.avgTimeOfStringArrayTypeWithArrayList            0  avgt    5   11,953 ± 0,309  ns/op
 * ColumnTypeBenchmark.avgTimeOfStringArrayTypeWithArrayList            2  avgt    5   25,266 ± 2,095  ns/op
 * ColumnTypeBenchmark.avgTimeOfStringArrayTypeWithArrayList            8  avgt    5   63,157 ± 2,787  ns/op
 * ColumnTypeBenchmark.avgTimeOfStringArrayTypeWithArrayList           16  avgt    5  113,761 ± 9,695  ns/op
 * ColumnTypeBenchmark.avgTimeOfStringArrayTypeWithArrayList           32  avgt    5  225,627 ± 8,457  ns/op
 *
 * Single call
 * Benchmark                                                                Mode  Cnt  Score   Error  Units
 * ColumnTypeBenchmark.SingleStringColumnTypeBenchmark.avgTimeOfStringType  avgt    5  6,303 ± 0,493  ns/op
 *
 * }
 * </pre>
 */

@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.NANOSECONDS)
@State(Scope.Benchmark)
@Fork(value = 1, jvmArgsAppend = {"-server"/*, "-XX:+PrintCompilation"*/})
@Warmup(iterations = 3)
@Measurement(iterations = 5)
public class ColumnTypeBenchmark {
    @Param({"0", "2", "8", "16", "32"})
    int arraySize = 1;

    private ColumnType ArrayStringType;
    private Object[] testArray;
    private ArrayList<String> testArrayList;

    // todo exclude META-INF/BenchmarkList from iceberg-inside libs
    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
            .include(ColumnTypeBenchmark.class.getSimpleName())
//            .include(SingleStringColumnTypeBenchmark.class.getSimpleName())
            .warmupIterations(3)
            .measurementIterations(5)
            .mode(Mode.AverageTime)
            .timeUnit(TimeUnit.NANOSECONDS)
            .forks(1)
            .build();

        new Runner(opt).run();
    }

    @Setup
    public void setup() {
        ArrayStringType = ColumnType.ArrayString;
        testArrayList = new ArrayList<>(arraySize);
        testArray = new Object[arraySize];

        for (int i = 0; i < arraySize; i++) {
            testArray[i] = "some value [" + i + "]";
            testArrayList.add("some value [" + i + "]");
        }
    }

    @Benchmark
    public boolean avgTimeOfStringArrayTypeWithArray() {
        return ArrayStringType.validate(testArray);
    }

    @Benchmark
    public boolean avgTimeOfStringArrayTypeWithArrayList() {
        return ArrayStringType.validate(testArrayList);
    }

//    @BenchmarkMode(Mode.AverageTime)
//    @OutputTimeUnit(TimeUnit.NANOSECONDS)
//    @State(Scope.Benchmark)
//    @Fork(value = 1, jvmArgsAppend = {"-server"/*, "-XX:+PrintCompilation"*/})
//    @Warmup(iterations = 3)
//    @Measurement(iterations = 5)
//    public static class SingleStringColumnTypeBenchmark {
//        private ColumnType StringType;
//        private String testStringVal;
//
//        @Setup
//        public void setup() {
//            StringType = ColumnType.String;
//            testStringVal = "some value[]";
//        }
//
//        /**
//         * Not actually baseline, just measure of single validation call
//         */
//        @Benchmark
//        public boolean avgTimeOfStringType() {
//            return StringType.validate(testStringVal);
//        }
//    }
}
