package ru.yandex.market.auth;

import org.apache.commons.io.IOUtils;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.DefaultRedirectStrategy;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author kukabara
 */
public class PassportClient implements InitializingBean {
    private static final Logger log = LogManager.getLogger();

    private static final Charset CHARSET = Charset.forName("UTF-8");
    private int timeoutSeconds = 60;
    private int maxConnections = 5;

    /**
     * https://doc.yandex-team.ru/Passport/passport-modes/concepts/location.xml
     */
    private String passportUrl;
    private HttpClient httpClient;

    @Override
    public void afterPropertiesSet() throws Exception {
        createHttpClient();
    }

    private void createHttpClient() {
        int timeoutMillis = (int) TimeUnit.SECONDS.toMillis(timeoutSeconds);

        RequestConfig requestConfig = RequestConfig.custom()
            .setConnectionRequestTimeout(timeoutMillis)
            .setConnectTimeout(timeoutMillis)
            .setSocketTimeout(timeoutMillis)
            .build();

        httpClient = HttpClientBuilder.create()
            .setMaxConnPerRoute(maxConnections)
            .setMaxConnTotal(maxConnections)
            .setDefaultRequestConfig(requestConfig)
            .setRedirectStrategy(new DefaultRedirectStrategy())
            .build();
    }

    /**
     * https://doc.yandex-team.ru/Passport/passport-modes/reference/auth.xml
     *
     * @param login
     * @param password
     */
    public List<String> getCookiesFromPassport(String login, String password) throws IOException {
        HttpPost request = new HttpPost(passportUrl + "/auth");
        List<NameValuePair> postParameters = new ArrayList<>();
        postParameters.add(new BasicNameValuePair("login", login));
        postParameters.add(new BasicNameValuePair("passwd", password));
        request.setEntity(new UrlEncodedFormEntity(postParameters, CHARSET));

        try {
            HttpResponse response = httpClient.execute(request);

            List<String> cookies = new ArrayList<>();
            Header[] cookieHeaders = response.getHeaders("Set-Cookie");
            // check that auth cookies was found
            Header[] sessions = new Header[2];
            for (Header header : cookieHeaders) {
                if (header.getValue().startsWith("Session_id=")) {
                    sessions[0] = header;
                } else if (header.getValue().startsWith("sessionid2=")) {
                    sessions[1] = header;
                }
                cookies.add(header.getValue().split(";")[0]);
            }
            if (sessions[0] == null || sessions[1] == null) {
                log.error("Can't find session cookies");
                String message = IOUtils.toString(response.getEntity().getContent());
                throw new IOException("Wrong http code:" + response.getStatusLine().getStatusCode() +
                    ", message: " + message);
            } else {
                log.debug("Successfully auth user " + login + " in " + passportUrl);
            }
            return cookies;
        } finally {
            request.releaseConnection();
        }
    }

    public void setPassportUrl(String passportUrl) {
        this.passportUrl = passportUrl;
    }
}
