package ru.yandex.market.clickhouse;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Joiner;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.io.ByteStreams;
import com.google.common.io.CountingInputStream;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HeaderElement;
import org.apache.http.HeaderElementIterator;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.conn.ConnectionKeepAliveStrategy;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.conn.PoolingClientConnectionManager;
import org.apache.http.impl.conn.SchemeRegistryFactory;
import org.apache.http.message.BasicHeaderElementIterator;
import org.apache.http.params.CoreConnectionPNames;
import org.apache.http.protocol.HTTP;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.jdbc.core.RowMapper;
import ru.yandex.clickhouse.LZ4EntityWrapper;
import ru.yandex.common.util.db.RowMappers;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

//import ru.yandex.metrika.util.collections.F;
//import ru.yandex.metrika.util.http.IpVersionPriorityResolver;
//import ru.yandex.metrika.util.io.FastByteArrayOutputStream;

/**
 * http клиент для работы с базой clickhouse
 * Для хитов используется БД hits, для лога изменений визитов - visit_log, для лога изменений истории посетителей - user_history_log.
 * <p>
 * curl 'http://localhost:8123/?database=hits&query='$(echo 'SHOW TABLES' | sed 's/ /%20/g')
 * curl 'http://localhost:8123/?database=hits&query='$(echo 'SELECT UniqID, EventTime FROM WatchLog_Chunk_2012071020170404110 LIMIT 10' | sed 's/ /%20/g')
 * wget -q -O- 'http://localhost:8123/?database=hits&query=SHOW TABLES'
 * wget -q -O- 'http://localhost:8123/?database=hits&query=SELECT UniqID, EventTime FROM WatchLog_Chunk_2012071020170404110 LIMIT 10'
 *
 * @author orantius
 * @version $Id$
 * @since 7/11/12
 */
@Deprecated
public class ClickhouseTemplate implements InitializingBean {
    static final Logger log = Logger.getLogger(ClickhouseTemplate.class);

    private static final int LOAD_CONFIGURATION_RETRY_INTERVAL_SECONDS = 5;

    private HttpClient httpClient;
    private ClickHouseSource db;
    private ObjectMapper objectMapper;
    private String beanName;
    private HttpConnectionProperties properties = new HttpConnectionProperties();
    private boolean replaceRunningQuery = false;

    public ClickhouseTemplate() {
    }


    public ClickhouseTemplate(String host, String database) {
        db = new ClickHouseSource(host, database);
        afterPropertiesSet();
    }

    public ClickhouseTemplate(ClickHouseSource db) {
        this.db = db;
        afterPropertiesSet();
    }


    public ClickhouseTemplate(ClickHouseSource db, HttpConnectionProperties properties) {
        this.db = db;
        this.properties = properties;
        afterPropertiesSet();

    }

    public void afterPropertiesSet() {
        PoolingClientConnectionManager conman =
            new PoolingClientConnectionManager(
                SchemeRegistryFactory.createDefault(),
                1, TimeUnit.MINUTES
            );
        conman.setDefaultMaxPerRoute(500);
        conman.setMaxTotal(1000);

        DefaultHttpClient client = new DefaultHttpClient(conman);
        client.setKeepAliveStrategy(createKeepAliveStrategy());
        httpClient = client;
        httpClient.getParams().setIntParameter(CoreConnectionPNames.SO_TIMEOUT, properties.getSocketTimeout());
        httpClient.getParams().setIntParameter(CoreConnectionPNames.CONNECTION_TIMEOUT, properties.getConnectionTimeout());
        httpClient.getParams().setIntParameter(CoreConnectionPNames.SOCKET_BUFFER_SIZE, properties.getApacheBufferSize());

        objectMapper = new ObjectMapper();
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);

        loadClusterConfiguration();
    }

    /**
     * Получаем из КХ информацию о серверах, которые входят в наш кластер.
     */
    private void loadClusterConfiguration() {
        String ddpApplyCluster = db.getDdlApplyCluster();
        if (db.isDistributed()) {
            log.info("Load configuration for cluster " + ddpApplyCluster);
            final List<String> hosts = new ArrayList<>();
            final Multimap<Integer, String> shard2Hosts = ArrayListMultimap.create();

            while (true) {
                for (String host : db.getHosts()) {
                    try {
                        query("select * from system.clusters where cluster = '" + ddpApplyCluster + "'",
                            host,
                            rs -> {
                                String hostName = rs.getString("host_name");
                                hosts.add(hostName);
                                shard2Hosts.put(rs.getInt("shard_num"), hostName);
                            });
                    } catch (Exception e) {
                        log.error("Unable to load cluster configuration", e);
                        continue;
                    }

                    db.setHosts(hosts);
                    db.setShard2Hosts(shard2Hosts);
                    log.info("Found hosts " + hosts + " for cluster " + ddpApplyCluster);
                    return;
                }
                try {
                    TimeUnit.SECONDS.sleep(LOAD_CONFIGURATION_RETRY_INTERVAL_SECONDS);
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
        }
    }

    private ConnectionKeepAliveStrategy createKeepAliveStrategy() {
        return new ConnectionKeepAliveStrategy() {

            @Override
            public long getKeepAliveDuration(HttpResponse response, HttpContext context) {
                // при ошибках keep-alive не всегда правильно работает, на всякий случай закроем коннекшн
                if (response.getStatusLine().getStatusCode() != HttpURLConnection.HTTP_OK) {
                    return -1;
                }
                HeaderElementIterator it = new BasicHeaderElementIterator(
                    response.headerIterator(HTTP.CONN_DIRECTIVE));
                while (it.hasNext()) {
                    HeaderElement he = it.nextElement();
                    String param = he.getName();
                    String value = he.getValue();
                    if (param != null && param.equalsIgnoreCase(HTTP.CONN_KEEP_ALIVE)) {
                        return properties.getKeepAliveTimeout();
                    }
                }
                return -1;
            }

        };
    }

    /**
     * Adding  FORMAT TabSeparatedWithNamesAndTypes if not added
     * Правильно реагирует на точку с запятой и втыкает формат только в селекты
     */
    public static String clickhousifySql(String sql, String format) {
        sql = sql.trim();
        if (StringUtils.startsWithIgnoreCase(sql, "SELECT")
            && !sql.replace(";", "").trim().endsWith(" TabSeparatedWithNamesAndTypes")
            && !sql.replace(";", "").trim().endsWith(" TabSeparated")
            && !sql.replace(";", "").trim().endsWith(" JSONCompact")) {
            if (sql.endsWith(";")) {
                sql = sql.substring(0, sql.length() - 1);
            }
            sql += " FORMAT " + format + ';';
        }
        return sql;
    }

    public static String clickhousifySql(String sql) {
        return clickhousifySql(sql, "TabSeparatedWithNamesAndTypes");
    }

    public int update(String sql) {
        List<String> hosts = db.getHosts();
        for (String host : hosts) {
            update(sql, host);
        }
        return hosts.size();
    }

    public int update(String sql, String host) {
        return update(sql, host, false);
    }

    public int update(String sql, String host, boolean ignoreDatabase) {
        InputStream inputStream = null;
        try {
            inputStream = getInputStream(sql, host, null, ignoreDatabase);
            ByteStreams.toByteArray(inputStream);
            return 1;
        } catch (ClickhouseException e) {
            throw e;
        } catch (Exception e) {
            throw new ClickhouseException("Exception executing sql:\n" + sql + "\n", e, host, db.getPort());
        } finally {
            IOUtils.closeQuietly(inputStream);
        }
    }

    public <T> List<T> query(String sql, RowMapper<T> rowMapper) {
        return query(sql, db.getHost(), rowMapper, null);
    }

    public <T> List<T> query(String sql, RowMapper<T> rowMapper, String queryId) {
        return query(sql, db.getHost(), rowMapper, queryId);
    }

    public <T> List<T> query(String sql, String host, RowMapper<T> rowMapper) {
        return query(sql, host, rowMapper, null);
    }

    public <T> List<T> query(String sql, String host, RowMapper<T> rowMapper, String queryId) {
        try (HttpResult httpResult = getHttpResult(sql, host, queryId)) {
            int count = 0;
            final List<T> ts = new ArrayList<>();
            while (httpResult.hasNext()) {
                HttpResultRow next = httpResult.getNext();
                ts.add(rowMapper.mapRow(next.split(), ++count));
            }

            // integer result is returned in the header without type
            if (ts.isEmpty() && httpResult.getTypes().length == 0) {
                HttpResultRow next = new HttpResultRow(httpResult, httpResult.getHeaderFragment());
                ts.add(rowMapper.mapRow(next.split(), ++count));
            }
            return ts;
        } catch (IOException e) {
            throw new ClickhouseConnectionException(e, host);
        } catch (Exception e) {
            throw new ClickhouseException("Exception executing sql:\n" + sql + "\n", e, db.getHost(), db.getPort());
        }
    }

    public void query(String sql, HttpRowCallbackHandler callbackHandler) {
        queryStreaming(sql, callbackHandler);
    }

    public void query(String sql, String host, HttpRowCallbackHandler callbackHandler) {
        queryStreaming(sql, host, callbackHandler);
    }

    public void query(String sql, HttpRowCallbackHandler callbackHandler, String queryId) {
        queryStreaming(sql, db.getHost(), callbackHandler, queryId);
    }


    public <T> T queryForObject(String sql, String host, RowMapper<T> rowMapper) {
        List<T> ts = query(sql, host, rowMapper);
        if (ts.size() == 1) {
            return ts.get(0);
        }
        throw new IllegalStateException(" " + ts.size() + " records instead of 1. query " + sql);
    }

    public <T> T queryForObject(String sql, RowMapper<T> rowMapper) {
        return queryForObject(sql, db.getHost(), rowMapper);
    }

    public <T> T queryForObjectNullable(String sql, RowMapper<T> rowMapper) {
        List<T> ts = query(sql, rowMapper);
        if (ts.isEmpty()) {
            return null;
        }
        if (ts.size() == 1) {
            return ts.get(0);
        }
        throw new IllegalStateException(" " + ts.size() + " records instead of 0 or 1. query " + sql);
    }

    public int queryForInt(String sql) throws ClickhouseException {
        return queryForInt(sql, db.getHost());
    }

    public int queryForInt(String sql, String host) throws ClickhouseException {
        List<Integer> result = query(sql, host, RowMappers.intAt(1));
        if (result.isEmpty()) {
            return 0;
        } else if (result.size() != 1) {
            throw new IllegalStateException("Wrong query result size: " + result.size() + " sql: " + sql);
        } else {
            return result.get(0);
        }
    }

    public String queryForString(String sql) throws ClickhouseException {
        return queryForString(sql, db.getHost());
    }

    public String queryForString(String sql, String host) throws ClickhouseException {
        List<String> result = query(sql, host, RowMappers.stringAt(1));
        if (result.isEmpty()) {
            return "";
        } else if (result.size() != 1) {
            throw new IllegalStateException("Wrong query result size: " + result.size() + " sql: " + sql);
        } else {
            return result.get(0);
        }
    }

    public long queryForLong(String sql) throws ClickhouseException {
        List<Long> result = query(sql, RowMappers.longAt(1));
        if (result.isEmpty()) {
            return 0;
        } else if (result.size() != 1) {
            throw new IllegalStateException("Wrong query result size: " + result.size() + " sql: " + sql);
        } else {
            return result.get(0);
        }
    }

    public void sendStream(InputStream content, String table, List<String> columns) throws IOException {
        // echo -ne '10\n11\n12\n' | POST 'http://localhost:8123/?query=INSERT INTO t FORMAT TabSeparated'
        HttpEntity entity = null;
        List<String> params = getParams(false, null);
        if (properties.isCompress()) {
            params.add("decompress=1");
            params.add("http_native_compression_disable_checksumming_on_decompress=1");
        }
        String query = Joiner.on('&').join(params) +
            "&query=INSERT INTO " + table +
            (columns == null || columns.isEmpty() ? "" : "(" + StringUtils.join(columns, ", ") + ")") +
            " FORMAT TabSeparated";
        try {
            URI uri = new URI(db.getScheme(), db.getAuthInfo(), db.getHost(), db.getPort(), "/", query, null);
            HttpPost httpPost = new HttpPost(uri);

            HttpEntity httpEntity = new InputStreamEntity(content, -1);
            if (properties.isCompress()) {
                httpEntity = new LZ4EntityWrapper(httpEntity, 1024 * 1024);
            }
            httpPost.setEntity(httpEntity);
            HttpResponse response = httpClient.execute(httpPost);
            entity = response.getEntity();
            if (response.getStatusLine().getStatusCode() != HttpURLConnection.HTTP_OK) {
                throw new ClickhouseException("response code = " + response.getStatusLine().getStatusCode(),
                    db.getHost(), db.getPort());
            }
        } catch (Exception e) {
            log.error("Error during connection to " + db, e);
            throw new IOException("Error during connection to " + db, e);
        } finally {
            if (entity != null) {
                EntityUtils.consumeQuietly(entity);
            }
        }
    }

    public void sendStream(InputStream content, String table) throws IOException {
        sendStream(content, table, null);
    }

    public void setDb(ClickHouseSource db) {
        this.db = db;
    }

    public <T> Iterable<T> queryStreaming(String s, RowMapper<T> rowMapper) {
        return new IterableHttpResult<>(this, s, rowMapper, null, null);
    }


    private InputStream getInputStream(String sql) throws IOException {
        return getInputStream(sql, db.getHost());
    }

    private InputStream getInputStream(String sql, String host) throws IOException {
        return getInputStream(sql, host, null, false);
    }

    private InputStream getInputStream(String sql, String host, String queryId, boolean ignoreDatabase)
        throws IOException {

        sql = clickhousifySql(sql);
        log.debug(sql);
        URI uri = null;
        try {
            String query = Joiner.on('&').join(getParams(ignoreDatabase, queryId));
            uri = new URI(db.getScheme(), db.getAuthInfo(), host, db.getPort(), "/", query, null);
        } catch (URISyntaxException e) {
            log.error("", e);
        }
        log.debug("Request url:" + uri);
        HttpPost post = new HttpPost(uri);
        post.setEntity(new StringEntity(sql, StandardCharsets.UTF_8));
        HttpEntity entity = null;
        InputStream is = null;
        try {
            HttpResponse response = httpClient.execute(post);

            entity = response.getEntity();
            if (response.getStatusLine().getStatusCode() != HttpURLConnection.HTTP_OK) {
                String errorMessage = null;
                try {
                    errorMessage = readError(entity);
                } catch (IOException e) {
                    errorMessage = "error while readFile response " + e.getMessage();
                }
                EntityUtils.consumeQuietly(entity);
                throw ClickHouseErrors.getSpecificException(errorMessage, new ClickhouseException(
                    "Response code " + response.getStatusLine().getStatusCode() +
                        " response: " + errorMessage, host, db.getPort()
                ));
            }
            if (entity.isStreaming()) {
                is = entity.getContent();
            } else {
                FastByteArrayOutputStream baos = new FastByteArrayOutputStream();
                entity.writeTo(baos);
                is = baos.convertToInputStream();
            }
            return is;
        } catch (IOException e) {
            EntityUtils.consumeQuietly(entity);
            try {
                if (is != null) {
                    is.close();
                }
            } catch (IOException ignored) {
            }
            throw e;
        }
    }

    private String readError(HttpEntity entity) throws IOException {
        InputStream inputStream = entity.getContent();
        if (properties.isCompress()) {
            inputStream = new ClickhouseLZ4Stream(inputStream);
        }
        return IOUtils.toString(inputStream);
    }

    public HttpResult getHttpResult(String sql) throws IOException {
        CountingInputStream is = new CountingInputStream(getInputStream(sql));
        return new HttpResult(properties.isCompress() ? new ClickhouseLZ4Stream(is) : is, is, properties.getBufferSize());
    }

    public HttpResult getHttpResult(String sql, String host) throws IOException {
        CountingInputStream is = new CountingInputStream(getInputStream(sql, host));
        return new HttpResult(properties.isCompress() ? new ClickhouseLZ4Stream(is) : is, is, properties.getBufferSize());
    }

    public HttpResult getHttpResult(String sql, String host, String queryId) throws IOException {
        CountingInputStream is = new CountingInputStream(getInputStream(sql, host, queryId, false));
        return new HttpResult(properties.isCompress() ? new ClickhouseLZ4Stream(is) : is, is, properties.getBufferSize());
    }


    public List<String> getParams(boolean ignoreDatabase, String queryId) {
        List<String> params = new ArrayList<>();
        //в clickhouse бывают таблички без базы (т.е. в базе default)
        if (!StringUtils.isBlank(db.getDb()) && !ignoreDatabase) {
            params.add("database=" + db.getDb());
        }
        if (properties.isCompress()) {
            params.add("compress=1");
        }
        // нам всегда нужны min и max в ответе
        params.add("extremes=1");
        if (!StringUtils.isEmpty(properties.getProfile())) {
            params.add("profile=" + properties.getProfile());
        } else {
            if (properties.getMaxThreads() != null) {
                params.add("max_threads=" + properties.getMaxThreads());
            }
            // да, там в секундах
            long maxExecutionTimeSeconds = TimeUnit.MILLISECONDS.toSeconds(
                properties.getSocketTimeout() + properties.getDataTransferTimeout());
            params.add("max_execution_time=" + maxExecutionTimeSeconds);
            if (properties.getMaxBlockSize() != null) {
                params.add("max_block_size=" + properties.getMaxBlockSize());
            }
        }
        //в кликхаус иногда бывает user
        if (properties.getUser() != null) {
            params.add("user=" + properties.getUser());
        }
        if (queryId != null) {
            params.add("query_id=" + queryId);
        }
        if (replaceRunningQuery) {
            params.add("replace_running_query=1");
        }
        return params;
    }


    public void ping() {
        queryForInt("SELECT 1");
    }

    public int queryStreaming(String sql, final HttpRowCallbackHandler callbackHandler) {
        return queryStreaming(sql, db.getHost(), callbackHandler, null);
    }

    public int queryStreaming(String sql, String host, final HttpRowCallbackHandler callbackHandler) {
        return queryStreaming(sql, host, callbackHandler, null);
    }

    public int queryStreaming(String sql, String host, final HttpRowCallbackHandler callbackHandler, String queryId) {
        try (HttpResult httpResult = getHttpResult(sql, host, queryId)) {
            long start = System.currentTimeMillis();
            int count = 0;
            while (httpResult.hasNext()) {
                count++;
                HttpResultRow next = httpResult.getNext();
                callbackHandler.processRow(next.split());
            }
            long bytesUncompressed = httpResult.getReadBytes();
            long bytesCompressed = httpResult.getRecievedBytes();
            long time = System.currentTimeMillis() - start;
            long byteSpeed = time > 0 ? bytesCompressed / time : Long.MAX_VALUE;
            long rowsSpeed = time > 0 ? count / time : Long.MAX_VALUE;
            log.debug(
                "query " + sql +
                    ", bytes Uncompressed " + bytesUncompressed +
                    ", bytes Compressed " + bytesCompressed +
                    ", speed: " + byteSpeed + " bytes per ms, " + rowsSpeed + " rows per ms"
            );
            return count;
        } catch (IOException e) {
            throw new ClickhouseConnectionException("Exception executing sql:\n" + sql + "\n", e, host);
        } catch (Exception e) {
            throw new ClickhouseException("Exception executing sql:\n" + sql + "\n", e, db.getHost(), db.getPort());
        }
    }


    public ClickHouseSource getDb() {
        return db;
    }

    public HttpConnectionProperties getProperties() {
        return properties;
    }

    public void setProperties(HttpConnectionProperties properties) {
        this.properties = properties;
    }

    public String getBeanName() {
        return beanName;
    }


    public String getDbName() {
        return db.getDb();
    }


    public void setBeanName(String beanName) {
        this.beanName = beanName;
    }

    public void setSocketTimeoutSeconds(int socketTimeoutSeconds) {
        properties.setSocketTimeoutSeconds(socketTimeoutSeconds);
    }

    public void setDataTransferTimeoutSeconds(int dataTransferTimeoutSeconds) {
        properties.setDataTransferTimeoutSeconds(dataTransferTimeoutSeconds);
    }

    public void setReplaceRunningQuery(boolean replaceRunningQuery) {
        this.replaceRunningQuery = replaceRunningQuery;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        ClickhouseTemplate that = (ClickhouseTemplate) o;

        return db.equals(that.db);

    }


    public int hashCode() {
        return db.hashCode();
    }


    public String toString() {
        return "HttpTemplateImpl{" +
            "db=" + db +
            '}';
    }
}
