package ru.yandex.market.clickhouse.ddl;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 28/03/2018
 */
public class ClickHouseCluster {
    private final String name;
    private final List<Server> servers;
    private final Map<String, Server> hostToServer;

    public ClickHouseCluster(String name, List<Server> servers) {
        this.name = name;
        this.servers = Collections.unmodifiableList(servers);
        hostToServer = servers.stream().collect(Collectors.toMap(Server::getHost, Function.identity()));
    }

    public String getName() {
        return name;
    }

    public List<Server> getServers() {
        return servers;
    }

    public Server getServer(String host) {
        return hostToServer.get(host);
    }

    public List<String> getAllHosts() {
        return servers.stream().map(Server::getHost).collect(Collectors.toList());
    }

    public static class Server {
        private final String host;
        private final int shardNumber;
        private final int replicaNumber;

        public Server(String host, int shardNumber, int replicaNumber) {
            this.host = host;
            this.shardNumber = shardNumber;
            this.replicaNumber = replicaNumber;
        }

        public String getHost() {
            return host;
        }

        public int getShardNumber() {
            return shardNumber;
        }

        public int getReplicaNumber() {
            return replicaNumber;
        }


        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Server server = (Server) o;
            return shardNumber == server.shardNumber &&
                replicaNumber == server.replicaNumber &&
                Objects.equals(host, server.host);
        }

        @Override
        public int hashCode() {
            return Objects.hash(host, shardNumber, replicaNumber);
        }

        @Override
        public String toString() {
            return "Server{" +
                "host='" + host + '\'' +
                ", shardNumber=" + shardNumber +
                ", replicaNumber=" + replicaNumber +
                '}';
        }
    }
}
