package ru.yandex.market.clickhouse.ddl;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.jdbc.core.JdbcTemplate;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 28/03/2018
 */
public class ClickHouseHostDdlDao {

    private static final String IP_ADDRESS_NOT_ALLOWED_ERROR_CODE = "195";
    private static final String USER_DEFAULT_IS_NOT_ALLOWED_TO_CONNECT_ERROR = "User default is not allowed to connect";

    private static final Logger log = LogManager.getLogger();

    private final ClickHouseCluster.Server server;
    private final JdbcTemplate jdbcTemplate;

    public ClickHouseHostDdlDao(ClickHouseCluster.Server server, JdbcTemplate jdbcTemplate) {
        this.server = server;
        this.jdbcTemplate = jdbcTemplate;
    }

    public String getHost() {
        return server.getHost();
    }

    public ClickHouseCluster.Server getServer() {
        return server;
    }

    private Optional<String> getTableDdl(String fullTableName) {
        boolean tableExists = jdbcTemplate.queryForObject("EXISTS TABLE " + fullTableName, Boolean.class);
        if (!tableExists) {
            return Optional.empty();
        }
        return Optional.of(jdbcTemplate.queryForObject("SHOW CREATE TABLE " + fullTableName, String.class));
    }

    public ServerQueryResult applyQuery(ClickHouseDdlService.QuerySupplier querySupplier) {
        try {
            List<String> queries = querySupplier.get(jdbcTemplate, server);
            for (String query : queries) {
                log.info("Execution on server {} query: {}", server, query);
                jdbcTemplate.update(query);
            }
        } catch (Exception e) {
            ServerQueryResult serverQueryResult = new ServerQueryResult(server, e);
            String error = serverQueryResult.getErrorString();
            String errorMessage = "Exception while execution query on host " + server.getHost();
                /* Skip the exception stacktrace,
                startrek ticket for an investigation - https://st.yandex-team.ru/CLICKHOUSE-4229 */
            if (Objects.equals(IP_ADDRESS_NOT_ALLOWED_ERROR_CODE, error) ||
                error.contains(USER_DEFAULT_IS_NOT_ALLOWED_TO_CONNECT_ERROR)) {
                log.warn(errorMessage);
            } else {
                log.warn(errorMessage, e);
            }
            return serverQueryResult;
        }
        return new ServerQueryResult(server);
    }

    public void applyQuery(DdlQuery query) {
        try {
            log.info("Host {}. Applying query: {}", server.getHost(), query.getQueryString());
            jdbcTemplate.update(query.getQueryString());
        } catch (Exception e) {
            //TODO skip some errors  UpdateHostDDLTaskImpl:46
            log.warn("Host " + server.getHost() + ". Exception on execution query: " + query.getQueryString(), e);
            throw e;
        }
    }

    public DDL getDdl(ClickHouseTableDefinition tableDefinition) {
        return TableUtils.buildDDL(
            tableDefinition,
            getTableDdl(tableDefinition.getFullTableName()),
            server.getHost()
        );
    }

    public Optional<ClickHouseTableDefinition> getExistedTableDefinition(TableName tableName) {
        Optional<String> tableDdl = getTableDdl(tableName.getFullName());
        return tableDdl.map(TableUtils::parseDDL);
    }

    public JdbcTemplate getJdbcTemplate() {
        return jdbcTemplate;
    }
}
