package ru.yandex.market.clickhouse.ddl;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Locale;

/**
 * @author Tatiana Litvinenko <a href="mailto:tanlit@yandex-team.ru"></a>
 * @date 25.05.2015
 */
public class Column {
    private final DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd", Locale.US);
    private final DateFormat dateTimeFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", Locale.US);

    private final String name;
    private final ColumnTypeBase type;
    // Значение DEFAULT для поля в DDL КХ.
    private final String defaultExpr;

    // Значение, которое пишем в КХ, если поле не нашлось в логах.
    private String defaultValue;
    private Object defaultObject;
    private String codec;

    public Column(String name, ColumnTypeBase type) {
        this(name, type, null);
    }

    /**
     * @param name
     * @param type
     * @param defaultExpr строка с дефолтным значением. Должна конвертиться в требуемый тип.
     */
    public Column(String name, ColumnTypeBase type, String defaultExpr) {
        this(name, type, defaultExpr, false);
    }

    /**
     * @param name
     * @param type
     * @param defaultExpr
     * @param fromClickhouse флаг, что defaultExpr получен из DDL КХ
     */
    public Column(String name, ColumnTypeBase type, String defaultExpr, boolean fromClickhouse) {
        Preconditions.checkArgument(
                defaultExpr == null || !defaultExpr.equals("[]"),
                "Default expression cannot be equal to [] for arrays. " +
                        "Please, use functions such as emptyArrayString() for this purpose."
        );
        this.name = name;
        this.type = type;
        this.defaultExpr = defaultExpr;
        this.defaultValue = defaultExpr;
    }

    /**
     * Конструктор специально для исключительных ситуаций, когда defaultExpr != defaultValue, например для массивов.
     *
     * @param name
     * @param type
     * @param defaultExpr
     * @param defaultValue значение по умолчанию, котрое будет использовно в insert into
     */
    public Column(String name, ColumnTypeBase type, String defaultExpr, String defaultValue) {
        this.name = name;
        this.type = type;
        this.defaultExpr = defaultExpr;
        this.defaultValue = defaultValue;
    }

    /**
     * Конструктор специально для использования codec.
     *
     * @param name
     * @param type
     * @param defaultExpr
     * @param defaultValue значение по умолчанию, котрое будет использовно в insert into
     * @param codec Clickhouse column codec
     */
    public Column(String name, ColumnTypeBase type, String defaultExpr, String defaultValue, String codec) {
        this.name = name;
        this.type = type;
        this.defaultExpr = defaultExpr;
        this.defaultValue = defaultValue;
        this.codec = codec;
    }

    public String getName() {
        return name;
    }

    public ColumnTypeBase getType() {
        return type;
    }

    public void setDefaultValue(String defaultValue) {
        this.defaultValue = defaultValue;
        this.defaultObject = type.parseValue(this.defaultValue, type == ColumnType.Date ? dateFormat : dateTimeFormat);
    }

    public String getDefaultExpr() {
        return defaultExpr;
    }

    public Object getDefaultObject() {
        return defaultObject;
    }

    public String getDefaultValue() {
        return defaultValue;
    }

    public String getCodec() {
        return codec;
    }


    @Override
    public String toString() {
        return name + " " + this.getType().toClickhouseDDL() +
                (Strings.isNullOrEmpty(defaultExpr) ? "" : " DEFAULT " + defaultExpr) +
                (Strings.isNullOrEmpty(codec) ? "" : " CODEC(" + codec + ")");
    }

    public String getQuotedDll() {
        return "`" + name + "` " + getType().toClickhouseDDL() +
                (Strings.isNullOrEmpty(defaultExpr) ? "" : " DEFAULT " + defaultExpr) +
                (Strings.isNullOrEmpty(codec) ? "" : " CODEC(" + codec + ")");
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Column)) {
            return false;
        }

        Column column = (Column) o;

        if (name != null ? !name.equals(column.name) : column.name != null) {
            return false;
        }

        if (!type.equals(column.type)) {
            return false;
        }

        if (codec != null ? !codec.equals(column.codec) : column.codec != null) {
            return false;
        }

        if (!type.areDefaultExpressionsEquals(defaultExpr, column.defaultExpr)) {
            return false;
        }

        return true;
    }

    @Override
    public int hashCode() {
        int result = name != null ? name.hashCode() : 0;
        result = 31 * result + (type != null ? type.hashCode() : 0);
        result = 31 * result + (defaultExpr != null ? defaultExpr.hashCode() : 0);
        result = 31 * result + (codec != null ? codec.hashCode() : 0);
        return result;
    }
}
