package ru.yandex.market.clickhouse.ddl;

import com.google.common.base.Preconditions;
import com.google.common.primitives.UnsignedLong;
import ru.yandex.clickhouse.util.ClickHouseRowBinaryStream;

import java.io.IOException;
import java.text.DateFormat;
import java.text.ParseException;
import java.util.Collection;
import java.util.Date;
import java.util.Objects;

/**
 * @author Tatiana Litvinenko <a href="mailto:tanlit@yandex-team.ru"></a>
 * @date 25.05.2015
 */
public enum ColumnType implements ColumnTypeBase {
    UInt8, UInt16, UInt32, UInt64, Int8, Int16, Int32, Int64,
    Float32, Float64,
    String,
    Date, DateTime,
    ArrayUInt8(UInt8, ColumnWrapper.ARRAY), ArrayUInt16(UInt16, ColumnWrapper.ARRAY),
    ArrayUInt32(UInt32, ColumnWrapper.ARRAY), ArrayUInt64(UInt64, ColumnWrapper.ARRAY),
    ArrayInt8(Int8, ColumnWrapper.ARRAY), ArrayInt16(Int16, ColumnWrapper.ARRAY),
    ArrayInt32(Int32, ColumnWrapper.ARRAY), ArrayInt64(Int64, ColumnWrapper.ARRAY),
    ArrayFloat32(Float32, ColumnWrapper.ARRAY), ArrayFloat64(Float64, ColumnWrapper.ARRAY),
    ArrayString(String, ColumnWrapper.ARRAY),
    ArrayDateTime(DateTime, ColumnWrapper.ARRAY),
    NullableUInt8(UInt8, ColumnWrapper.NULLABLE), NullableUInt16(UInt16, ColumnWrapper.NULLABLE),
    NullableUInt32(UInt32, ColumnWrapper.NULLABLE), NullableUInt64(UInt64, ColumnWrapper.NULLABLE),
    NullableString(String, ColumnWrapper.NULLABLE);

    private static final long U_INT8_MAX = (1 << 8) - 1;
    private static final long U_INT16_MAX = (1 << 16) - 1;
    private static final long U_INT32_MAX = (1L << 32) - 1;
    private static final long MAX_DATE = 2177355600636L; //Fri Dec 31 00:00:00 FET 2038
    private static final long MIN_DATE = 0; //Thu Jan 01 00:00:00 FET 1970
    private static final long FLOAT_32_MIN = (long) Float.MIN_VALUE;
    private static final long FLOAT_32_MAX = (long) Float.MAX_VALUE;

    private final ColumnType subtype;
    private final String clickHouseName;
    private final boolean isNullable;
    private final boolean isArray;

    ColumnType() {
        clickHouseName = name();
        subtype = null;
        isNullable = false;
        isArray = false;
    }

    ColumnType(ColumnType subtype, ColumnWrapper wrapper) {
        this.subtype = subtype;
        if (wrapper == ColumnWrapper.ARRAY) {
            isArray = true;
            isNullable = false;
        } else if (wrapper == ColumnWrapper.NULLABLE) {
            isArray = false;
            isNullable = true;
        } else {
            throw new UnsupportedOperationException("Unknown column wrapper: " + wrapper);
        }

        clickHouseName = wrapper.getClickhouseFunction() + "(" + subtype.name() + ")";
    }

    public ColumnType getSubtype() {
        return subtype;
    }

    @Override
    public String toClickhouseDDL() {
        return clickHouseName;
    }

    @Override
    public boolean isEnum() {
        return false;
    }

    @Override
    public boolean isArray() {
        return isArray;
    }

    @Override
    public boolean isNullable() {
        return isNullable;
    }

    @Override
    public boolean canBeModifiedToAutomatically(ColumnTypeBase replacement) {
        return false;
    }

    @Override
    public boolean validate(Object value) {
        switch (this) {
            case String:
                return value != null;
            case Date:
            case DateTime:
                if (value instanceof Date) {
                    Date date = (Date) value;
                    return date.getTime() >= MIN_DATE && date.getTime() <= MAX_DATE;
                }
                return false;
            case Int8:
                return validateNumber(value, Byte.MIN_VALUE, Byte.MAX_VALUE);
            case Int16:
                return validateNumber(value, Short.MIN_VALUE, Short.MAX_VALUE);
            case Int32:
                return validateNumber(value, Integer.MIN_VALUE, Integer.MAX_VALUE);
            case Int64:
                return validateNumber(value, Long.MIN_VALUE, Long.MAX_VALUE);
            case UInt8:
                return validateNumber(value, 0, U_INT8_MAX);
            case UInt16:
                return validateNumber(value, 0, U_INT16_MAX);
            case UInt32:
                return validateNumber(value, 0, U_INT32_MAX);
            case UInt64:
                return value instanceof UnsignedLong || validateNumber(value, 0, Long.MAX_VALUE);
            case Float32:
                if (value instanceof Float || validateNumber(value, FLOAT_32_MIN, FLOAT_32_MAX)) {
                    return true;
                }
                if (value instanceof Double) {
                    Double doubleValue = (Double) value;
                    return doubleValue > Float.MIN_VALUE && doubleValue <= Float.MAX_VALUE;
                }
                return false;
            case Float64:
                if (value instanceof Double || value instanceof Float) {
                    return true;
                }
                return validateNumber(value, Long.MIN_VALUE, Long.MAX_VALUE);
            default:
                if (subtype != null) {
                    if (isArray()) {
                        return validateArrayValue(subtype, value);
                    }

                    if (isNullable()) {
                        return validateNullableValue(subtype, value);
                    }
                }
                throw new IllegalStateException("Unknown column type: " + this);
        }
    }

    private boolean validateNumber(Object value, long minValue, long maxValue) {
        if (value instanceof Long) {
            Long longValue = (Long) value;
            return longValue >= minValue && longValue <= maxValue;
        }
        if (value instanceof Integer) {
            Integer intValue = (Integer) value;
            return intValue >= minValue && intValue <= maxValue;
        }
        if (value instanceof Short) {
            Short shortValue = (Short) value;
            return shortValue >= minValue && shortValue <= maxValue;
        }
        if (value instanceof Byte) {
            Byte byteValue = (Byte) value;
            return byteValue >= minValue && byteValue <= maxValue;
        }
        if (value instanceof Boolean) {
            return true;
        }
        return false;
    }

    @Override
    public Object parseValue(String value, DateFormat dateFormat) {
        return parseValue(this, value, dateFormat);
    }

    private Object parseValue(ColumnType type, String value, DateFormat dateFormat) {
        switch (type) {
            case String:
                return value;
            case Date:
            case DateTime:
                return parseDate(value, dateFormat);
            case Int8:
            case UInt8:
            case Int16:
            case UInt16:
            case Int32:
                switch (value) {
                    case "false":
                    case "FALSE":
                        return 0;
                    case "true":
                    case "TRUE":
                        return 1;
                    default:
                        return Integer.valueOf(value);
                }
            case UInt32:
            case Int64:
                switch (value) {
                    case "false":
                    case "FALSE":
                        return 0;
                    case "true":
                    case "TRUE":
                        return 1;
                    default:
                        return Long.valueOf(value);
                }
            case UInt64:
                switch (value) {
                    case "false":
                    case "FALSE":
                        return 0;
                    case "true":
                    case "TRUE":
                        return 1;
                    default:
                        return UnsignedLong.valueOf(value);
                }
            case Float32:
                return Float.valueOf(value);
            case Float64:
                return Double.valueOf(value);
            default:
                if (type.getSubtype() != null) {
                    if (type.isArray()) {
                        return parseArrayValue(type, value, dateFormat);
                    }

                    if (isNullable()) {
                        return parseNullableValue(type, value, dateFormat);
                    }
                }

                throw new IllegalStateException("Unknown column type: " + type);
        }
    }

    private Object parseArrayValue(ColumnType type, String value, DateFormat dateFormat) {
        if (value.isEmpty()) {
            return new Object[0];
        }

        String[] split = value.split(",");
        Object[] l = new Object[split.length];
        if (split.length == 1 && split[0].equals("[]")) {
            l[0] = split[0];
            return l;
        }
        for (int i = 0; i < split.length; i++) {
            l[i] = parseValue(type.getSubtype(), split[i], dateFormat);
        }
        return l;
    }

    private Object parseNullableValue(ColumnType type, String value, DateFormat dateFormat) {
        if (value == null) {
            return null;
        }

        return parseValue(type.getSubtype(), value, dateFormat);
    }

    @Override
    public void writeTo(Object value, ClickHouseRowBinaryStream stream) throws IOException {
        switch (this) {
            case String:
                stream.writeString(value.toString());
                break;
            case Date:
                stream.writeDate((Date) value);
                break;
            case DateTime:
                stream.writeDateTime((Date) value);
                break;
            case Int8:
                stream.writeInt8(getInt(value));
                break;
            case UInt8:
                stream.writeUInt8(getInt(value));
                break;
            case Int16:
                stream.writeInt16(getInt(value));
                break;
            case UInt16:
                stream.writeUInt16(getInt(value));
                break;
            case Int32:
                stream.writeInt32(getInt(value));
                break;
            case UInt32:
                stream.writeUInt32(getLong(value));
                break;
            case Int64:
                stream.writeInt64(getLong(value));
                break;
            case UInt64:
                if (value instanceof UnsignedLong) {
                    stream.writeUInt64((UnsignedLong) value);
                } else {
                    stream.writeUInt64(getLong(value));
                }
                break;
            case Float32:
                stream.writeFloat32(getFloat(value));
                break;
            case Float64:
                stream.writeFloat64(getDouble(value));
                break;
            default:
                Preconditions.checkState(subtype != null, "Unknown column type: " + this);
                if (isArray()) {
                    writeArray(subtype, value, stream);
                } else if (isNullable()) {
                    if (value == null) {
                        stream.markNextNullable(true);
                    } else {
                        subtype.writeTo(value, stream);
                    }
                }
        }
    }

    @Override
    public void format(Object value, StringBuilder valueBuilder) {
        if (value instanceof Boolean) {
            valueBuilder.append((Boolean) value ? "1" : "0");
        } else if (value instanceof Date) {
            if (this == ColumnType.Date) {
                valueBuilder.append(dateFormatHolder.get().format((Date) value));
            } else {
                valueBuilder.append(dateTimeFormatHolder.get().format((Date) value));
            }
        } else if (value instanceof Double || value instanceof Float) {
            //toLowerCase т.к. кликхаус хочет видеть nan, а не NaN
            valueBuilder.append(value.toString().toLowerCase());
        } else if (value instanceof Object[]) {
            Preconditions.checkState(subtype != null, "Bad type %s for array", this);
            formatArray(subtype, (Object[]) value, valueBuilder);
        } else if (value instanceof Collection) {
            format(((Collection) value).toArray(), valueBuilder);
        } else {
            valueBuilder.append(CLICKHOUSE_ESCAPER.escape(value.toString()));
        }
    }

    @Override
    public boolean areDefaultExpressionsEquals(String defaultExpr, String otherDefaultExpr) {
        return Objects.equals(defaultExpr, otherDefaultExpr);
    }

    private float getFloat(Object value) {
        if (value instanceof Number) {
            return ((Number) value).floatValue();
        } else if (value instanceof Boolean) {
            return (Boolean) value ? 1f : 0f;
        }
        throw new IllegalStateException("Not a number. Class: " + value.getClass() + ", value:" + value.toString());
    }

    private double getDouble(Object value) {
        if (value instanceof Number) {
            return ((Number) value).doubleValue();
        } else if (value instanceof Boolean) {
            return (Boolean) value ? 1d : 0d;
        }
        throw new IllegalStateException("Not a number. Class: " + value.getClass() + ", value:" + value.toString());
    }

    private int getInt(Object value) {
        if (value instanceof Number) {
            return ((Number) value).intValue();
        } else if (value instanceof Boolean) {
            return (Boolean) value ? 1 : 0;
        }
        throw new IllegalStateException("Not a number. Class: " + value.getClass() + ", value:" + value.toString());
    }

    private long getLong(Object value) {
        if (value instanceof Number) {
            return ((Number) value).longValue();
        } else if (value instanceof Boolean) {
            return (Boolean) value ? 1L : 0L;
        }
        throw new IllegalStateException("Not a number. Class: " + value.getClass() + ", value:" + value.toString());
    }

    private Object parseDate(String value, DateFormat dateFormat) {
        try {
            return new Date(Long.parseLong(value));
        } catch (NumberFormatException e) {
            try {
                return dateFormat.parse(value);
            } catch (ParseException e1) {
                throw new RuntimeException("Failed to parse date", e1);
            }
        }
    }
}
