package ru.yandex.market.clickhouse.ddl;

import com.google.common.base.Preconditions;
import com.google.common.escape.Escaper;
import com.google.common.escape.Escapers;
import ru.yandex.clickhouse.util.ClickHouseRowBinaryStream;

import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.Locale;

/**
 * @author Anton Sukhonosenko <a href="mailto:algebraic@yandex-team.ru"></a>
 * @date 18/08/16
 */
public interface ColumnTypeBase {

    Escaper CLICKHOUSE_ESCAPER = Escapers.builder()
        .addEscape('\\', "\\\\")
        .addEscape('\'', "\\'")
        .addEscape('\t', "\\t")
        .addEscape('\n', "\\n")
        .addEscape('\b', "\\b")
        .addEscape('\f', "\\f")
        .addEscape('\0', "\\r0")
        .build();

    ThreadLocal<DateFormat> dateFormatHolder = ThreadLocal.withInitial(
        () -> new SimpleDateFormat("yyyy-MM-dd", Locale.US));
    ThreadLocal<DateFormat> dateTimeFormatHolder = ThreadLocal.withInitial(
        () -> new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", Locale.US));

    boolean validate(Object o);

    Object parseValue(String value, DateFormat dateFormat);

    String toClickhouseDDL();

    boolean isEnum();

    boolean isArray();

    boolean isNullable();

    String name();

    boolean canBeModifiedToAutomatically(ColumnTypeBase replacement);

    void writeTo(Object value, ClickHouseRowBinaryStream stream) throws IOException;

    default void writeArray(ColumnTypeBase subtype, Object value, ClickHouseRowBinaryStream stream) throws IOException {
        Preconditions.checkState(
            value instanceof Object[] || value instanceof Collection,
            "Value is not array or collection: " + value);

        Object[] values;
        if (value instanceof Object[]) {
            values = (Object[]) value;
        } else {
            values = ((Collection<?>) value).toArray();
        }
        //Just values, without length
        for (Object element : values) {
            subtype.writeTo(element, stream);
        }
    }

    void format(Object value, StringBuilder valueBuilder);

    default void formatItem(Object value, StringBuilder valueBuilder) {
        if (value instanceof Date || value instanceof String) {
            valueBuilder.append('\'');
            format(value, valueBuilder);
            valueBuilder.append('\'');
        } else {
            format(value, valueBuilder);
        }
    }

    default boolean validateArrayValue(ColumnTypeBase subtype, Object value) {
        if (value instanceof Object[]) {
            for (Object o : (Object[]) value) {
                if (!subtype.validate(o)) {
                    return false;
                }
            }
            return true;
        } else if (value instanceof Collection) {
            for (Object o : (Collection) value) {
                if (!subtype.validate(o)) {
                    return false;
                }
            }
            return true;
        } else {
            return false;
        }
    }

    default boolean validateNullableValue(ColumnTypeBase subtype, Object value) {
        if (value == null) {
            return true;
        }

        return subtype.validate(value);
    }

    default void formatArray(ColumnTypeBase subtype, Object[] value, StringBuilder valueBuilder) {
        valueBuilder.append("[");
        for (int i = 0; i < value.length; ++i) {
            if (i > 0) {
                valueBuilder.append(",");
            }
            subtype.formatItem(value[i], valueBuilder);
        }
        valueBuilder.append("]");
    }

    boolean areDefaultExpressionsEquals(String defaultExpr, String otherDefaultExpr);
}
