package ru.yandex.market.clickhouse.ddl;

import com.google.common.base.Preconditions;
import ru.yandex.clickhouse.except.ClickHouseErrorCode;
import ru.yandex.clickhouse.except.ClickHouseException;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 05/06/2018
 */
public class ServerQueryResult {

    private final ClickHouseCluster.Server server;
    private final Status status;
    private final Exception exception;
    private final ClickHouseErrorCode errorCode;

    public ServerQueryResult(ClickHouseCluster.Server server) {
        this.server = server;
        status = Status.OK;
        exception = null;
        errorCode = null;
    }

    public ServerQueryResult(ClickHouseCluster.Server server, Exception exception) {
        this.server = server;
        if (!(exception instanceof ClickHouseException) && exception.getCause() instanceof ClickHouseException) {
            exception = (Exception) exception.getCause();
        }
        if (exception instanceof ClickHouseException) {
            ClickHouseException clickHouseException = (ClickHouseException) exception;
            ClickHouseErrorCode code = ClickHouseErrorCode.fromCode(clickHouseException.getErrorCode());
            this.errorCode = code != null ? code : ClickHouseErrorCode.UNKNOWN_EXCEPTION;
            this.status = Status.CLICKHOUSE_EXCEPTION;
        } else {
            this.errorCode = null;
            this.status = Status.RUNTIME_EXCEPTION;
        }
        this.exception = exception;
    }

    public ClickHouseCluster.Server getServer() {
        return server;
    }

    public Status getStatus() {
        return status;
    }

    public Exception getException() {
        return exception;
    }

    public ClickHouseErrorCode getErrorCode() {
        return errorCode;
    }

    public String getErrorString() {
        Preconditions.checkState(status != Status.OK);
        if (status == Status.CLICKHOUSE_EXCEPTION) {
            return errorCode.toString();
        } else {
            return exception.toString();
        }
    }

    public enum Status {
        OK,
        CLICKHOUSE_EXCEPTION,
        RUNTIME_EXCEPTION
    }
}
