package ru.yandex.market.clickhouse.ddl;

import com.google.common.collect.ListMultimap;
import com.google.common.collect.Multimaps;
import ru.yandex.market.monitoring.MonitoringStatus;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 02/04/2018
 */
public class TableDdlState {
    private final MonitoringStatus globalStatus;
    private final List<ShardState> shardStates;
    private final Map<String, ServerState> hostToState = new HashMap<>();

    public TableDdlState(MonitoringStatus globalStatus, List<ShardState> shardStates) {
        this.globalStatus = globalStatus;
        this.shardStates = shardStates;

        for (ShardState shardState : shardStates) {
            for (ServerState serverState : shardState.getServerStates()) {
                hostToState.put(serverState.getHost(), serverState);
            }
        }
    }

    public ServerState getServerState(String host){
        return hostToState.get(host);
    }

    public MonitoringStatus getGlobalStatus() {
        return globalStatus;
    }

    public List<ShardState> getShardStates() {
        return shardStates;
    }

    public static class ShardState {

        private final int shardNumber;
        private final MonitoringStatus shardStatus;
        private final List<ServerState> serverStates;

        public ShardState(int shardNumber, MonitoringStatus shardStatus, List<ServerState> serverStates) {
            this.shardNumber = shardNumber;
            this.shardStatus = shardStatus;
            this.serverStates = serverStates;
        }

        public int getShardNumber() {
            return shardNumber;
        }

        public MonitoringStatus getShardStatus() {
            return shardStatus;
        }

        public List<ServerState> getServerStates() {
            return serverStates;
        }

        @Override
        public String toString() {
            return "ShardState{" +
                "shardNumber=" + shardNumber +
                ", shardStatus=" + shardStatus +
                ", serverStates=" + serverStates +
                '}';
        }
    }

    public static class ServerState {
        private final ClickHouseCluster.Server server;
        private final MonitoringStatus replicaStatus;
        private final DDL ddl;
        private final String message;

        public ServerState(ClickHouseCluster.Server server, MonitoringStatus replicaStatus, DDL ddl, String message) {
            this.server = server;
            this.replicaStatus = replicaStatus;
            this.ddl = ddl;
            this.message = message;
        }

        public String getHost() {
            return server.getHost();
        }

        public int getShardNumber() {
            return server.getShardNumber();
        }

        public int getReplicaNumber() {
            return server.getReplicaNumber();
        }

        public MonitoringStatus getReplicaStatus() {
            return replicaStatus;
        }

        public DDL getDdl() {
            return ddl;
        }

        @Override
        public String toString() {
            return "ServerState{" +
                "server=" + server +
                ", replicaStatus=" + replicaStatus +
                ", ddl=" + ddl +
                ", message='" + message + '\'' +
                '}';
        }
    }

    public static TableDdlState create(List<ServerState> serverStates) {
        ListMultimap<Integer, ServerState> serversByShard = Multimaps.index(serverStates, ServerState::getShardNumber);
        List<ShardState> shardStates = new ArrayList<>();
        for (Integer shardNumber : serversByShard.keySet()) {
            List<ServerState> shardServerStates = serversByShard.get(shardNumber);
            MonitoringStatus shardStatus = shardServerStates.stream()
                .map(ServerState::getReplicaStatus)
                .min(Comparator.naturalOrder())
                .get();
            shardStates.add(new ShardState(shardNumber, shardStatus, shardServerStates));
        }

        MonitoringStatus globalStatus = shardStates.stream()
            .map(ShardState::getShardStatus)
            .max(Comparator.naturalOrder())
            .get();
        return new TableDdlState(globalStatus, shardStates);
    }

    @Override
    public String toString() {
        return "TableDdlState{" +
            "globalStatus=" + globalStatus +
            ", shardStates=" + shardStates +
            '}';
    }
}
