package ru.yandex.market.clickhouse.ddl;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import ru.yandex.market.clickhouse.ddl.engine.DistributedEngine;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.clickhouse.ddl.engine.ReplacingMergeTree;
import ru.yandex.market.clickhouse.ddl.engine.ReplicatedMergeTree;

import static ru.yandex.market.clickhouse.ddl.DdlQueryType.ADD_COLUMN;
import static ru.yandex.market.clickhouse.ddl.DdlQueryType.CREATE_DATABASE;
import static ru.yandex.market.clickhouse.ddl.DdlQueryType.CREATE_TABLE;
import static ru.yandex.market.clickhouse.ddl.DdlQueryType.MODIFY_COLUMN;

/**
 * @author Tatiana Litvinenko <a href="mailto:tanlit@yandex-team.ru"></a>
 * @date 29.05.2015
 */
public class TableUtils {
    private static final Logger log = LogManager.getLogger();

    protected static final String CREATE_TABLE_FORMAT = "CREATE TABLE IF NOT EXISTS %s ( %s ) ENGINE = %s";
    protected static final String ADD_COLUMN_FORMAT = "ALTER TABLE %s ADD COLUMN %s %s";
    protected static final String MODIFY_COLUMN_FORMAT = "ALTER TABLE %s MODIFY COLUMN %s %s";
    protected static final String DROP_COLUMN_FORMAT = "ALTER TABLE %s DROP COLUMN %s";
    protected static final String DEFAULT_CODEC = "LZ4";

    private static final Pattern COLUMN_PATTERN = Pattern.compile(
            "`?(?<name>\\w+\\.?\\w+)`?\\s+" +
                    "(?<type>(LowCardinality|Nullable|Array\\()?Enum(8|16)\\(.+?\\)\\)?|[\\w\\(\\)]+)" +
                    "(\\s+DEFAULT\\s+(?<default>CAST\\(.+?\\)|.+?))?" +
                    "(\\s+CODEC\\((?<codec>.+?)\\))?" +
                    "(,  |, `|$)"
    );

    private static final Pattern OLD_MERGE_TREE_PATTERN = Pattern.compile(
            "^(?<dateField>\\w+), \\(?(?<primaryKey>[\\w, ]+)\\)?, (?<indexGranularity>\\d+)$"
    );

    private static final Pattern OLD_REPLICATED_MERGE_TREE_PATTERN = Pattern.compile(
            "^'(?<zkPath>[\\w{}\\/\\.]+)', '(?<zkReplica>[\\w{}\\/\\.]+)', (?<dateField>\\w+), " +
                    "\\(?(?<primaryKey>[\\w, ]+)\\)?, (?<indexGranularity>\\d+)$"
    );

    private static final Pattern ENGINE_PATTERN = Pattern.compile(
            "^(?<engineName>[a-zA-Z]+)\\((?<engineParams>.*)\\)( (?<engineLine>PARTITION BY .*))?$"
    );

    private static final Pattern NEW_MERGE_TREE_LINE_PATTERN = Pattern.compile(
            "^PARTITION BY (?<partitionBy>.+) ORDER BY \\(?(?<orderBy>.+?)\\)?" +
                    "( SAMPLE BY (?<sampleBy>.+))? SETTINGS (?<settings>.+)$"
    );

    private static final Pattern NEW_REPLICATED_MERGE_TREE_PARAMS_PATTERN = Pattern.compile(
            "^'(?<zkPath>[\\w{}/.]+)', '(?<zkReplica>[\\w{}/.]+)'(, (?<engineParams>\\w+.+))?$"
    );

    private static final Pattern DISTRIBUTED_PARAMS_PATTERN = Pattern.compile(
            "^'?(?<cluster>\\w+)'?, '?(?<database>\\w+)'?, '?(?<table>\\w+)'?(, (?<shardingKey>\\w+.+))?$"
    );

    private static final Splitter PK_SPLITTER = Splitter.on(',').trimResults();

    private TableUtils() {
    }

    public static ClickHouseTableDefinition parseDDL(String ddl) {
        ddl = ddl.replace('\n', ' ');
        String fullTableName = parseFullTableName(ddl);
        int engineIndex = ddl.indexOf("ENGINE =");
        String columnsDDL = ddl.substring(ddl.indexOf("(") + 1, engineIndex);
        columnsDDL = columnsDDL.substring(0, columnsDDL.lastIndexOf(")"));
        List<Column> tableColumns = parseColumns(columnsDDL);
        EngineType engine = parseEngine(ddl.substring(engineIndex + "ENGINE =".length()).trim());
        return new ClickHouseTableDefinitionImpl(fullTableName, tableColumns, engine);
    }

    private static String parseFullTableName(String ddl) {
        return ddl.replace("CREATE TABLE ", "").split(" ")[0];
    }

    @VisibleForTesting
    static EngineType parseEngine(String engineString) {
        Matcher matcher = ENGINE_PATTERN.matcher(engineString);
        Preconditions.checkArgument(matcher.matches(), "Can't parse engine string: %s", engineString);

        String name = matcher.group("engineName");
        String params = matcher.group("engineParams");
        String line = matcher.group("engineLine");

        if (name.contains("MergeTree") && line != null) {
            return parseNewMergeTree(name, params, line);
        }
        switch (name) {
            case "MergeTree":
                matcher = OLD_MERGE_TREE_PATTERN.matcher(params);
                Preconditions.checkArgument(matcher.matches(), "Can't parse old MergeTree params: %s", params);
                return parseOldMergeTree(matcher);
            case "ReplicatedMergeTree":
                matcher = OLD_REPLICATED_MERGE_TREE_PATTERN.matcher(params);
                Preconditions.checkArgument(
                        matcher.matches(),
                        "Can't parse old ReplicatedMergeTree params: %s", params
                );
                return new ReplicatedMergeTree(
                        parseOldMergeTree(matcher),
                        matcher.group("zkPath"),
                        matcher.group("zkReplica")
                );
            case "Distributed":
                matcher = DISTRIBUTED_PARAMS_PATTERN.matcher(params);
                Preconditions.checkArgument(matcher.matches(), "Can't parse Distributed params: %s", params);
                return new DistributedEngine(
                        matcher.group("cluster"),
                        matcher.group("database"),
                        matcher.group("table"),
                        matcher.group("shardingKey")
                );
            default:
                throw new UnsupportedOperationException("Unsupported engine: " + name);
        }
    }

    private static MergeTree parseOldMergeTree(Matcher mergeTreeFamilyMatcher) {
        return MergeTree.fromOldDefinition(
                mergeTreeFamilyMatcher.group("dateField"),
                PK_SPLITTER.splitToList(mergeTreeFamilyMatcher.group("primaryKey")),
                Integer.parseInt(mergeTreeFamilyMatcher.group("indexGranularity"))
        );
    }

    private static EngineType parseNewMergeTree(String engineName, String engineParams, String engineLine) {
        MergeTree mergeTree = parseNewMergeTreeSettings(engineLine);
        engineParams = engineParams.trim();
        boolean isReplicated = engineName.startsWith(ReplicatedMergeTree.REPLICATED_PREFIX);
        String zookeeperTablePath = null;
        String zookeeperReplicaName = null;
        if (isReplicated) {
            engineName = engineName.substring(ReplicatedMergeTree.REPLICATED_PREFIX.length());
            Matcher matcher = NEW_REPLICATED_MERGE_TREE_PARAMS_PATTERN.matcher(engineParams);
            Preconditions.checkState(
                matcher.matches(), "Cannon parse ReplicatedMergeTree params: %s", engineParams
            );
            engineParams = matcher.group("engineParams");
            if (engineParams != null) {
                engineParams = engineParams.trim();
            }
            zookeeperTablePath = matcher.group("zkPath");
            zookeeperReplicaName = matcher.group("zkReplica");
        }
        switch (engineName) {
            case "MergeTree":
                Preconditions.checkState(
                    engineParams == null || engineParams.isEmpty(),
                    "Unsupported MergeTree params: %s", engineParams
                );
                break;
            case "ReplacingMergeTree":
                mergeTree = new ReplacingMergeTree(mergeTree, engineParams);
                break;
            default:
                throw new UnsupportedOperationException("Unsupported MergeTree family engine: " + engineLine);
        }
        if (isReplicated) {
            return new ReplicatedMergeTree(mergeTree, zookeeperTablePath, zookeeperReplicaName);
        } else {
            return mergeTree;
        }
    }

    @VisibleForTesting
    static MergeTree parseNewMergeTreeSettings(String mergeTreeLine) {
        Matcher matcher = NEW_MERGE_TREE_LINE_PATTERN.matcher(mergeTreeLine);
        Preconditions.checkArgument(matcher.matches(), "Can't parse merge tree settings string: %s", mergeTreeLine);
        Map<String, String> engineSettings = Splitter.on(',')
                .trimResults()
                .withKeyValueSeparator('=')
                .split(matcher.group("settings"));

        int indexGranularity = MergeTree.DEFAULT_INDEX_GRANULARITY;
        for (Map.Entry<String, String> settingEntry : engineSettings.entrySet()) {
            String key = settingEntry.getKey().trim();
            String value = settingEntry.getValue().trim();
            switch (key) {
                case "index_granularity":
                    indexGranularity = Integer.parseInt(value);
                    break;
                default:
                    throw new UnsupportedOperationException(
                            String.format(
                                    "Unsupported merge tree param '%s' in merge tree line: %s", key, mergeTreeLine
                            )
                    );
            }
        }

        List<String> orderBy = PK_SPLITTER.splitToList(matcher.group("orderBy"));
        return new MergeTree(
                matcher.group("partitionBy"),
                orderBy,
                matcher.group("sampleBy"),
                indexGranularity
        );
    }

    @VisibleForTesting
    static List<Column> parseColumns(String columns) {
        columns = columns.trim();
        Matcher matcher = COLUMN_PATTERN.matcher(columns);
        List<Column> tableColumns = new ArrayList<>();

        while (matcher.find()) {
            String name = matcher.group("name");
            ColumnTypeBase type = ColumnTypeUtils.fromClickhouseDDL(matcher.group("type"));
            String defaultExpr = unwrapDefaultExpr(type, matcher.group("default"));
            String codec = matcher.group("codec");
            tableColumns.add(new Column(name, type, defaultExpr, defaultExpr, codec));
        }
        return tableColumns;
    }

    /**
     * В КХ есть особенность, что дефолтные значения Enum и некоторых других типов он оборачивает в CAST.
     * Например при Enum8('UNSTABLE' = 0, 'TESTING' = 1, 'PRESTABLE' = 2, 'STABLE' = 3, 'UNKNOWN' = 4)
     * указать в defaultExpr 'UNSTABLE', КХ обернет это в
     * CAST('UNKNOWN', 'Enum8(\'UNSTABLE\' = 0, \'TESTING\' = 1, \'PRESTABLE\' = 2, \'STABLE\' = 3, \'UNKNOWN\' = 4)')
     * Мы убираем каст, что бы не применять DDL повторно
     * Еже один аспект - не очень страшно, если у нас не получиться применить выцепить настоящие значение,
     * главное не выцепить только часть.
     *
     * @param type
     * @param defaultExpr
     * @return
     */
    @VisibleForTesting
    static String unwrapDefaultExpr(ColumnTypeBase type, String defaultExpr) {
        if (Strings.isNullOrEmpty(defaultExpr)) {
            return defaultExpr;
        }

        // Удаляем "CAST(" из начала и ", 'Тип')" из конца
        String expectedStart = "CAST(";
        String expectedEnd = ", '" + type.toClickhouseDDL().replace("'", "\\'") + "')";
        if (defaultExpr.startsWith(expectedStart) && defaultExpr.endsWith(expectedEnd)) {
            return StringUtils.removeEnd(
                    StringUtils.removeStart(defaultExpr, expectedStart),
                    expectedEnd
            );
        }
        return defaultExpr;
    }

    /**
     * @param expectedTableDefinition
     * @param existingTableDdl
     * @return набор изменений, который нужно применить на хосте, для получения требуемой таблицы.
     */
    public static DDL buildDDL(ClickHouseTableDefinition expectedTableDefinition,
                               Optional<String> existingTableDdl, String host) {
        DDL tableDdl = new DDL(host);

        if (!existingTableDdl.isPresent()) {
            tableDdl.addQuery(new DdlQuery(
                    CREATE_DATABASE, "CREATE DATABASE IF NOT EXISTS " + expectedTableDefinition.getDatabaseName()
            ));

            String columnsList = expectedTableDefinition.getColumns().stream()
                    .map(Column::getQuotedDll)
                    .collect(Collectors.joining(", "));
            String engineDDL = expectedTableDefinition.getEngine().createEngineDDL();
            tableDdl.addQuery(
                    new DdlQuery(
                            CREATE_TABLE,
                            String.format(
                                    CREATE_TABLE_FORMAT, expectedTableDefinition.getFullTableName(), columnsList, engineDDL
                            )
                    )
            );
        } else {
            checkTable(expectedTableDefinition, existingTableDdl.get(), tableDdl);
        }
        return tableDdl;
    }

    private static void checkTable(ClickHouseTableDefinition expectedTable, String existedTableDDL, DDL tableDdl) {
        ClickHouseTableDefinition existedTable = parseDDL(existedTableDDL);

        if (!expectedTable.getEngine().equals(existedTable.getEngine())) {
            tableDdl.addError(
                    "Table engine has been changed for table " + expectedTable.getFullTableName() + "! " +
                            "Current: " + existedTable.getEngine() + ", new: " + expectedTable.getEngine()
            );
        }

        Iterator<Column> it;
        boolean exists;

        List<Column> existedColumns = new ArrayList<>(existedTable.getColumns());
        if (!expectedTable.getColumns().equals(existedColumns)) {

            for (Column column : expectedTable.getColumns()) {
                exists = false;
                String columnName = column.getName();

                it = existedColumns.iterator();
                while (it.hasNext()) {
                    Column existedColumn = it.next();

                    if (columnName.equals(existedColumn.getName())) {
                        if (!column.getType().equals(existedColumn.getType())) {
                            log.info(
                                    "Planned DDL: Will MODIFY column '{}' in table '{}' because its type is wrong" +
                                            " in ClickHouse (expected '{}', actual '{}')",
                                    columnName, expectedTable.getFullTableName(),
                                    column.getType(), existedColumn.getType()
                            );
                            if (existedColumn.getType().canBeModifiedToAutomatically(column.getType())) {
                                addModifyColumnTypeQuery(tableDdl, expectedTable.getFullTableName(), column);
                            } else {
                                addModifyColumnTypeManualQuery(tableDdl, expectedTable.getFullTableName(), column);
                            }
                        } else if (different(column.getDefaultExpr(), existedColumn.getDefaultExpr())) {
                            log.info(
                                    "Planned DDL: Will MODIFY column '{}' in table '{}' because its default value is wrong" +
                                            " in ClickHouse (expected '{}', actual '{}')",
                                    columnName, expectedTable.getFullTableName(),
                                    column.getDefaultExpr(), existedColumn.getDefaultExpr()
                            );
                            tableDdl.addQuery(
                                    new DdlQuery(
                                            MODIFY_COLUMN,
                                            String.format(
                                                    MODIFY_COLUMN_FORMAT, expectedTable.getFullTableName(), columnName,
                                                    column.getDefaultExpr() == null ? column.getType().toClickhouseDDL() :
                                                            column.getType().toClickhouseDDL() + " DEFAULT " + column.getDefaultExpr()
                                            )
                                    )
                            );
                        } else if (differentCodec(column.getCodec(), existedColumn.getCodec())) {
                            log.info(
                                    "Planned DDL: Will MODIFY column '{}' in table '{}' because its codec is wrong" +
                                            " in ClickHouse (expected '{}', actual '{}')",
                                    columnName, expectedTable.getFullTableName(),
                                    column.getCodec(), existedColumn.getCodec()
                            );
                            tableDdl.addManualQuery(
                                    new DdlQuery(
                                            MODIFY_COLUMN,
                                            String.format(
                                                    MODIFY_COLUMN_FORMAT, expectedTable.getFullTableName(), columnName,
                                                    "CODEC(" + (column.getCodec() == null ? DEFAULT_CODEC : column.getCodec()) + ")"
                                            )
                                    )
                            );
                        }
                        exists = true;
                        it.remove();
                        break;
                    }
                }

                if (!exists) {
                    log.info(
                            "Planned DDL: Will ADD column '{}' to table '{}'",
                            columnName, expectedTable.getFullTableName()
                    );
                    tableDdl.addQuery(
                            new DdlQuery(
                                    ADD_COLUMN,
                                    String.format(
                                            ADD_COLUMN_FORMAT,
                                            expectedTable.getFullTableName(),
                                            columnName,
                                            column.getType().toClickhouseDDL() +
                                                    (column.getDefaultExpr() == null ? "" : " DEFAULT " + column.getDefaultExpr())
                                    )
                            )
                    );
                }
            }

            for (Column column : existedColumns) {
                String columnName = column.getName();
                log.info(
                        "Planned DDL: Will DROP column '{}' from table '{}'",
                        columnName, expectedTable.getFullTableName()
                );
                if (expectedTable.getEngine().containsColumn(columnName)) {
                    tableDdl.addError(
                            "Can't drop column " + columnName +
                                    " cause it appears in the engine: " + expectedTable.getEngine()
                    );
                } else {
                    tableDdl.addManualQuery(
                            new DdlQuery(
                                    DdlQueryType.DROP_COLUMN,
                                    String.format(DROP_COLUMN_FORMAT, expectedTable.getFullTableName(), columnName)
                            )
                    );
                }
            }
        }
    }

    private static void addModifyColumnTypeManualQuery(DDL tableDDL, String tableName, Column column) {
        String query = String.format(
                MODIFY_COLUMN_FORMAT,
                tableName,
                column.getName(),
                column.getType().toClickhouseDDL()
        );

        tableDDL.addManualQuery(new DdlQuery(DdlQueryType.MODIFY_COLUMN, query));
    }

    private static void addModifyColumnTypeQuery(DDL tableDDL, String tableName, Column column) {
        String query = String.format(
                MODIFY_COLUMN_FORMAT,
                tableName,
                column.getName(),
                column.getType().toClickhouseDDL()
        );

        tableDDL.addQuery(new DdlQuery(MODIFY_COLUMN, query));
    }

    private static boolean different(String expr1, String expr2) {
        if (expr1 == null) {
            return expr2 != null;
        }
        return !expr1.equals(expr2);
    }

    private static boolean differentCodec(String codec1, String codec2) {
        if (!different(codec1, codec2)) {
            return false;
        }

        if (codec1 == null && codec2.equals(DEFAULT_CODEC)) {
            return false;
        }

        return true;
    }
}
