package ru.yandex.market.clickhouse.ddl.engine;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;

/**
 * @author Tatiana Litvinenko <a href="mailto:tanlit@yandex-team.ru"></a>
 * @date 28.05.2015
 */
public class MergeTree implements IMergeTreeEngineType {

    public static final int DEFAULT_INDEX_GRANULARITY = 8192;

    private final String partitionBy;
    private final List<String> orderBy;
    private final String sampleBy;
    private final int indexGranularity;

    public MergeTree(MergeTree mergeTree) {
        this(mergeTree.partitionBy, mergeTree.orderBy, mergeTree.sampleBy, mergeTree.indexGranularity);
    }

    public MergeTree(String partitionBy, List<String> orderBy, String sampleBy, int indexGranularity) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(partitionBy), "Empty partitionBy");
        Preconditions.checkArgument(orderBy != null && !orderBy.isEmpty(), "Empty orderBy");
        if (sampleBy != null) {
            Preconditions.checkArgument(
                orderBy.contains(sampleBy),
                "Sampling expression '%s' must be present in the primary key (order by): %s",
                sampleBy, orderBy
            );
        }
        this.partitionBy = partitionBy;
        this.orderBy = Collections.unmodifiableList(orderBy);
        this.sampleBy = sampleBy;
        this.indexGranularity = indexGranularity;
    }

    public MergeTree(String partitionBy, List<String> orderBy, String sampleBy) {
        this(partitionBy, orderBy, sampleBy, DEFAULT_INDEX_GRANULARITY);
    }

    public MergeTree(String partitionBy, List<String> orderBy) {
        this(partitionBy, orderBy, null, DEFAULT_INDEX_GRANULARITY);
    }

    public static MergeTree fromOldDefinition(String dateField, List<String> primaryKey, int indexGranularity) {
        return new MergeTree("toYYYYMM(" + dateField + ")", primaryKey, null, indexGranularity);
    }

    public static MergeTree fromOldDefinition(String dateField, List<String> primaryKey) {
        return new MergeTree("toYYYYMM(" + dateField + ")", primaryKey);
    }

    public static MergeTree fromOldDefinition(String dateField, String... primaryKey) {
        return new MergeTree("toYYYYMM(" + dateField + ")", Arrays.asList(primaryKey));
    }

    @Override
    public boolean containsColumn(String columnName) {
        //Достаточно грубая функция, можно лучше, но стоит ли?
        if (orderBy.contains(columnName) || partitionBy.equals(columnName)) {
            return true;
        }
        String columnInFunction = "(" + columnName + ")";
        if (partitionBy.contains(columnInFunction)) {
            return true;
        }
        for (String orderByEntry : orderBy) {
            if (orderByEntry.contains(columnInFunction)) {
                return true;
            }
        }
        return false;
    }

    protected String getMergeTreeSettingsString() {
        if (sampleBy != null) {
            return String.format(
                "PARTITION BY %s ORDER BY (%s) SAMPLE BY %s SETTINGS index_granularity = %d",
                partitionBy, String.join(", ", orderBy), sampleBy, indexGranularity
            );
        } else {
            return String.format(
                "PARTITION BY %s ORDER BY (%s) SETTINGS index_granularity = %d",
                partitionBy, String.join(", ", orderBy), indexGranularity
            );
        }
    }

    public String getPartitionBy() {
        return partitionBy;
    }

    public List<String> getOrderBy() {
        return orderBy;
    }

    public String getSampleBy() {
        return sampleBy;
    }

    @Override
    public String createEngineDDL() {
        return String.format("%s(%s) %s", getEngineTypeName(), getEngineParams(), getMergeTreeSettingsString());
    }

    @Override
    public EngineType replicated(String database, String table, String zkTablePrefix) {
        return new ReplicatedMergeTree(
            this,
            database,
            table,
            zkTablePrefix
        );
    }

    public String getEngineTypeName() {
        return "MergeTree";
    }

    public String getEngineParams() {
        return "";
    }

    public int getIndexGranularity() {
        return indexGranularity;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        MergeTree mergeTree = (MergeTree) o;
        return indexGranularity == mergeTree.indexGranularity &&
            Objects.equals(partitionBy, mergeTree.partitionBy) &&
            Objects.equals(orderBy, mergeTree.orderBy) &&
            Objects.equals(sampleBy, mergeTree.sampleBy);
    }

    @Override
    public int hashCode() {
        return Objects.hash(partitionBy, orderBy, sampleBy, indexGranularity);
    }

    @Override
    public String toString() {
        return "MergeTree{" +
            "partitionBy='" + partitionBy + '\'' +
            ", orderBy=" + orderBy +
            ", sampleBy='" + sampleBy + '\'' +
            ", indexGranularity=" + indexGranularity +
            '}';
    }
}
