package ru.yandex.market.clickhouse.ddl.engine;

import java.util.List;
import java.util.Objects;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;

/**
 * @author kukabara
 */
public class ReplicatedMergeTree implements IMergeTreeEngineType {
    private static final String DEFAULT_REPLICA_NAME = "{replica}";
    public static final String REPLICATED_PREFIX = "Replicated";

    private final MergeTree mergeTree;
    private final String zookeeperTablePath;
    private final String zookeeperReplicaName;

    public ReplicatedMergeTree(MergeTree mergeTree, String zookeeperTablePath, String zookeeperReplicaName) {
        this.mergeTree = mergeTree;
        this.zookeeperTablePath = zookeeperTablePath;
        this.zookeeperReplicaName = zookeeperReplicaName;
        validate();
    }

    public ReplicatedMergeTree(MergeTree mergeTree, String database, String table, String zkTablePrefix) {
        this.mergeTree = mergeTree;
        this.zookeeperTablePath = String.format("%s/%s.%s", zkTablePrefix, database, table);
        this.zookeeperReplicaName = DEFAULT_REPLICA_NAME;
    }

    @Deprecated
    public ReplicatedMergeTree(String partitionBy, List<String> orderBy, String sampleBy, int indexGranularity,
                               String zookeeperTablePath, String zookeeperReplicaName) {
        this(new MergeTree(partitionBy, orderBy, sampleBy, indexGranularity), zookeeperTablePath, zookeeperReplicaName);
    }

    @Deprecated
    public ReplicatedMergeTree(String partitionBy, List<String> orderBy, String sampleBy,
                               String zookeeperTablePath, String zookeeperReplicaName) {
        this(new MergeTree(partitionBy, orderBy, sampleBy), zookeeperTablePath, zookeeperReplicaName);
    }

    @Deprecated
    public ReplicatedMergeTree(String partitionBy, List<String> orderBy, String zookeeperTablePath,
                               String zookeeperReplicaName) {
        this (new MergeTree(partitionBy, orderBy), zookeeperTablePath, zookeeperReplicaName);
    }

    private void validate() {
        Preconditions.checkArgument(mergeTree != null);
        Preconditions.checkArgument(!Strings.isNullOrEmpty(zookeeperTablePath));
        Preconditions.checkArgument(!Strings.isNullOrEmpty(zookeeperReplicaName));
        Preconditions.checkArgument(
            !zookeeperTablePath.contains("'"), "zookeeperTablePath should't contain single commas: %s",
            zookeeperTablePath
        );
        Preconditions.checkArgument(
            !zookeeperReplicaName.contains("'"), "zookeeperReplicaName should't contain single commas: %s",
            zookeeperReplicaName
        );
    }

    @Override
    public boolean containsColumn(String columnName) {
        return mergeTree.containsColumn(columnName);
    }

    @Override
    public String createEngineDDL() {
        String engineParams = mergeTree.getEngineParams();
        if (engineParams == null || engineParams.isEmpty()) {
            return String.format("%s%s('%s', '%s') %s",
                REPLICATED_PREFIX,
                mergeTree.getEngineTypeName(),
                zookeeperTablePath, zookeeperReplicaName,
                mergeTree.getMergeTreeSettingsString());
        } else {
            return String.format("%s%s('%s', '%s', %s) %s",
                REPLICATED_PREFIX,
                mergeTree.getEngineTypeName(),
                zookeeperTablePath, zookeeperReplicaName,
                engineParams,
                mergeTree.getMergeTreeSettingsString());
        }
    }

    @Override
    public String getPartitionBy() {
        return mergeTree.getPartitionBy();
    }

    @Override
    public EngineType replicated(String database, String table, String zkTablePrefix) {
        return this;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof ReplicatedMergeTree)) {
            return false;
        }
        ReplicatedMergeTree that = (ReplicatedMergeTree) o;
        return mergeTree.equals(that.mergeTree) &&
            zookeeperTablePath.equals(that.zookeeperTablePath) &&
            zookeeperReplicaName.equals(that.zookeeperReplicaName);
    }

    @Override
    public int hashCode() {
        return Objects.hash(mergeTree, zookeeperTablePath, zookeeperReplicaName);
    }
}
