package ru.yandex.market.clickhouse.ddl.enums;

import com.google.common.base.Preconditions;
import ru.yandex.clickhouse.util.ClickHouseRowBinaryStream;
import ru.yandex.market.clickhouse.ddl.ColumnTypeBase;

import java.io.IOException;
import java.text.DateFormat;
import java.util.Collection;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author Anton Sukhonosenko <a href="mailto:algebraic@yandex-team.ru"></a>
 * @date 16.10.17
 */
public class EnumArrayColumnType implements ColumnTypeBase {
    private static final Pattern DEFAULT_EXPR_PATTERN = Pattern.compile("CAST\\(\\s*(?<value>.*?)\\s+AS Array\\(Enum.*?\\)");

    private final EnumColumnType subtype;
    private final String clickhouseName;

    private EnumArrayColumnType(EnumColumnType subtype) {
        Preconditions.checkArgument(subtype != null, "subtype must be not null");
        this.subtype = subtype;
        this.clickhouseName = "Array(" + subtype.toClickhouseDDL() + ")";
    }

    public static EnumArrayColumnType enum8Array(Class<?> enumClass) {
        return new EnumArrayColumnType(EnumColumnType.enum8(enumClass));
    }

    public static EnumArrayColumnType enum16Array(Class<?> enumClass) {
        return new EnumArrayColumnType(EnumColumnType.enum16(enumClass));
    }

    @Override
    public boolean validate(Object value) {
        return validateArrayValue(subtype, value);
    }

    @Override
    public Object parseValue(String value, DateFormat dateFormat) {
        String[] split = value.split(",");
        Object[] l = new Object[split.length];
        for (int i = 0; i < split.length; i++) {
            l[i] = subtype.parseValue(split[i], dateFormat);
        }
        return l;
    }

    @Override
    public String toClickhouseDDL() {
        return clickhouseName;
    }

    @Override
    public boolean isEnum() {
        return true;
    }

    @Override
    public boolean isArray() {
        return true;
    }

    @Override
    public boolean isNullable() {
        return false;
    }

    @Override
    public String name() {
        return clickhouseName;
    }

    @Override
    public boolean canBeModifiedToAutomatically(ColumnTypeBase replacement) {
        return false;
    }

    @Override
    public void writeTo(Object value, ClickHouseRowBinaryStream stream) throws IOException {
        writeArray(subtype, value, stream);
    }

    @Override
    public void format(Object value, StringBuilder valueBuilder) {
        if (value instanceof Object[]) {
            formatArray(subtype, (Object[]) value, valueBuilder);
        } else if (value instanceof Collection) {
            format(((Collection) value).toArray(), valueBuilder);
        } else {
            throw new UnsupportedOperationException("Unknown type of value: " + value.getClass());
        }
    }

    @Override
    public boolean areDefaultExpressionsEquals(String defaultExpr, String otherDefaultExpr) {
        return Objects.equals(normalizeDefaultExpression(defaultExpr), normalizeDefaultExpression(otherDefaultExpr));
    }

    private String normalizeDefaultExpression(String defaultExpr) {
        if (defaultExpr == null) {
            return null;
        }

        Matcher matcher = DEFAULT_EXPR_PATTERN.matcher(defaultExpr);
        if (matcher.matches()) {
            return matcher.group("value");
        }

        return defaultExpr;
    }

    public static ColumnTypeBase fromClickhouseDDL(String name) {
        EnumColumnType subtype;
        if (name.startsWith("Array(Enum8")) {
            subtype = new EnumColumnType(EnumType.Enum8, parseEnumConstants(name));
        } else if (name.startsWith("Array(Enum16")) {
            subtype = new EnumColumnType(EnumType.Enum16, parseEnumConstants(name));
        } else {
            throw new UnsupportedOperationException("Unable to parse enum array from DDL: " + name);
        }
        return new EnumArrayColumnType(subtype);
    }

    private static EnumConstants parseEnumConstants(String name) {
        String[] parts = name.split("[\\(\\)]");
        Preconditions.checkState(
            parts.length == 3,
            "Unexpected parts count after split by '(' and ')', expected 5, got %s, expression = %s",
            parts.length, name
        );

        return EnumConstants.fromCommaSeparatedConstants(parts[2]);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        EnumArrayColumnType that = (EnumArrayColumnType) o;
        return Objects.equals(subtype, that.subtype);
    }

    @Override
    public int hashCode() {
        return Objects.hash(subtype);
    }
}
