package ru.yandex.market.clickhouse.ddl.enums;

import ru.yandex.clickhouse.util.ClickHouseRowBinaryStream;
import ru.yandex.market.clickhouse.ddl.ColumnTypeBase;

import java.io.IOException;
import java.text.DateFormat;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * @author Anton Sukhonosenko <a href="mailto:algebraic@yandex-team.ru"></a>
 * @date 18/08/16
 */
public class EnumColumnType implements ColumnTypeBase {
    private static final Pattern DEFAULT_EXPR_PATTERN = Pattern.compile("CAST\\(\\s*(?<value>.*?)\\s+AS Enum.*?\\)");

    private final EnumType type;
    private final EnumConstants enumConstants;
    private final String clickHouseDdl;

    EnumColumnType(EnumType type, Class<?> enumClass) {
        this(type, getEnumConstants(type, enumClass));
    }

    EnumColumnType(EnumType type, EnumConstants enumConstants) {
        this.type = type;
        this.enumConstants = enumConstants;
        this.clickHouseDdl = type.toString() + "(" + enumConstants.toString() + ")";
    }

    public static EnumColumnType enum8(Class<?> enumClass) {
        return new EnumColumnType(EnumType.Enum8, enumClass);
    }

    public static EnumColumnType enum16(Class<?> enumClass) {
        return new EnumColumnType(EnumType.Enum16, enumClass);
    }

    @Override
    public boolean validate(Object o) {
        if (o instanceof Enum<?>) {
            Enum<?> enumValue = (Enum<?>) o;
            return enumConstants.containsValue(enumValue.name());
        }

        if (o instanceof String) {
            String stringValue = (String) o;
            return enumConstants.containsValue(stringValue);
        }

        throw new IllegalArgumentException("Unknown type of argument: " + o.getClass());
    }

    @Override
    public Object parseValue(String value, DateFormat dateFormat) {
        if (!this.validate(value)) {
            throw new IllegalArgumentException("Invalid value: " + value);
        }

        return value;
    }

    @Override
    public String toClickhouseDDL() {
        return clickHouseDdl;
    }

    @Override
    public boolean isEnum() {
        return true;
    }

    @Override
    public boolean isNullable() {
        return false;
    }

    @Override
    public boolean isArray() {
        return false;
    }

    @Override
    public String name() {
        return type.name();
    }

    @Override
    public boolean canBeModifiedToAutomatically(ColumnTypeBase replacement) {
        if (replacement == null || !(replacement instanceof EnumColumnType)) {
            return false;
        }

        EnumColumnType enumReplacement = (EnumColumnType) replacement;

        if (!type.equals(enumReplacement.type)) {
            return false;
        }

        return enumConstants.isSubsetOf(enumReplacement.enumConstants);
    }

    @Override
    public void writeTo(Object value, ClickHouseRowBinaryStream stream) throws IOException {
        int enumPosition;
        if (value instanceof Enum<?>) {
            Enum<?> enumValue = (Enum<?>) value;
            enumPosition = enumValue.ordinal();
        } else if (value instanceof String) {
            String stringValue = (String) value;
            enumPosition = enumConstants.getPosition(stringValue);
        } else {
            throw new IllegalArgumentException("Unknown type of argument: " + value.getClass());
        }
        switch (type) {
            case Enum8:
                stream.writeInt8(enumPosition);
                break;
            case Enum16:
                stream.writeInt16(enumPosition);
                break;
            default:
                throw new IllegalStateException();
        }
    }

    @Override
    public void format(Object value, StringBuilder valueBuilder) {
        if (value instanceof Enum<?>) {
            Enum<?> enumValue = (Enum<?>) value;
            valueBuilder.append(CLICKHOUSE_ESCAPER.escape(enumValue.name()));
        } else {
            valueBuilder.append(CLICKHOUSE_ESCAPER.escape(value.toString()));
        }
    }

    @Override
    public boolean areDefaultExpressionsEquals(String defaultExpr, String otherDefaultExpr) {
        return Objects.equals(normalizeDefaultExpression(defaultExpr), normalizeDefaultExpression(otherDefaultExpr));
    }

    private String normalizeDefaultExpression(String defaultExpr) {
        if (defaultExpr == null) {
            return null;
        }

        Matcher matcher = DEFAULT_EXPR_PATTERN.matcher(defaultExpr);
        if (matcher.matches()) {
            return matcher.group("value");
        }

        return defaultExpr;
    }

    public static EnumColumnType fromClickhouseDDL(String name) {
        EnumConstants enumConstants = EnumConstants.fromClickhouseDDL(name);
        return name.startsWith("Enum8")
            ? new EnumColumnType(EnumType.Enum8, enumConstants)
            : new EnumColumnType(EnumType.Enum16, enumConstants);
    }

    private static EnumConstants getEnumConstants(EnumType type, Class<?> enumClass) {
        List<? extends Enum<?>> enumMembers = Arrays.stream(enumClass.getEnumConstants())
            .map(c -> (Enum<?>) c)
            .collect(Collectors.toList());

        if (type.equals(EnumType.Enum8)) {
            if (!enumMembers.stream().allMatch(c -> c.ordinal() >= Byte.MIN_VALUE && c.ordinal() <= Byte.MAX_VALUE)) {
                throw new IllegalArgumentException(String.format("Too big enum %s to fit it into Enum8", enumClass));
            }
        }

        if (type.equals(EnumType.Enum16)) {
            if (!enumMembers.stream().allMatch(c -> c.ordinal() >= Short.MIN_VALUE && c.ordinal() <= Short.MAX_VALUE)) {
                throw new IllegalArgumentException(String.format("Too big enum %s to fit it into Enum16", enumClass));
            }
        }

        return new EnumConstants(enumMembers);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof EnumColumnType)) {
            return false;
        }

        EnumColumnType that = (EnumColumnType) obj;
        if (!this.type.equals(that.type)) {
            return false;
        }

        return this.enumConstants.equals(that.enumConstants);
    }
}
