package ru.yandex.market.clickhouse.ddl.enums;

import com.google.common.base.Preconditions;

import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Collector;
import java.util.stream.Collectors;

/**
 * @author Anton Sukhonosenko <a href="mailto:algebraic@yandex-team.ru"></a>
 * @date 18/08/16
 */
class EnumConstants {
    // TreeMap, чтобы взять values() у двух различных коллекций и сравнить значение к значению
    private final TreeMap<Integer, EnumConstant> constants;
    private final Map<String, Integer> positions;

    EnumConstants(List<? extends Enum<?>> enumMembers) {
        this(enumMembers.stream().map(c -> new EnumConstant(c.name(), c.ordinal())).collect(toTreeMap()));
    }

    private EnumConstants(TreeMap<Integer, EnumConstant> enumConstants) {
        constants = enumConstants;
        positions = constants.entrySet().stream().collect(Collectors.toMap(
            entry -> entry.getValue().getName(), Map.Entry::getKey
        ));
    }

    static EnumConstants fromClickhouseDDL(String clickhouseDDLTypeInfo) {
        String[] parts = clickhouseDDLTypeInfo.split("[\\(\\)]");
        return fromCommaSeparatedConstants(parts[1]);
    }

    static EnumConstants fromCommaSeparatedConstants(String constantsString) {
        String[] constants = constantsString.split(",");
        TreeMap<Integer, EnumConstant> enumConstants = Arrays.stream(constants)
            .map(EnumConstant::fromClickhouseDDL)
            .collect(toTreeMap());

        return new EnumConstants(enumConstants);
    }

    public boolean containsValue(String name) {
        return positions.containsKey(name);
    }

    public int getPosition(String name) {
        Integer position = positions.get(name);
        Preconditions.checkArgument(position != null, "No value '%s' in enum", name);
        return position;
    }

    @Override
    public String toString() {
        return constants.values().stream().map(EnumConstant::toString).collect(Collectors.joining(", "));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }

        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        EnumConstants that = (EnumConstants) o;

        if (this.constants.size() != that.constants.size()) {
            return false;
        }

        Iterator<EnumConstant> it = that.constants.values().iterator();
        for (EnumConstant thisConstant : this.constants.values()) {
            EnumConstant thatConstant = it.next();
            if (!thisConstant.equals(thatConstant)) {
                return false;
            }
        }

        return true;
    }

    @Override
    public int hashCode() {
        return constants.values().stream().map(EnumConstant::getOrdinal).reduce(0, (acc, o) -> acc ^ o);
    }

    public boolean isSubsetOf(EnumConstants enumConstants) {
        return constants.values().stream()
            .allMatch(c -> {
                EnumConstant thatConstant = enumConstants.constants.getOrDefault(c.getOrdinal(), null);
                return thatConstant != null && thatConstant.equals(c);
            });
    }

    private static Collector<EnumConstant, ?, TreeMap<Integer, EnumConstant>> toTreeMap() {
        return Collectors.toMap(
            EnumConstant::getOrdinal,
            c -> c,
            (u, v) -> {
                throw new IllegalStateException(String.format("Duplicate key %s", u));
            },
            TreeMap::new
        );
    }
}
