package ru.yandex.market.conductor;

import com.google.gson.FieldNamingPolicy;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.springframework.beans.factory.InitializingBean;
import ru.yandex.market.conductor.model.ConductorTicket;

import java.io.IOException;
import java.lang.reflect.Type;
import java.net.URISyntaxException;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * https://beta.wiki.yandex-team.ru/conductor/api/
 * https://beta.wiki.yandex-team.ru/Conductor/API/handles
 * Актуальный список методов https://c.yandex-team.ru/api/https://c.yandex-team.ru/api/
 *
 * @author kukabara
 */
public class ConductorClient implements InitializingBean {
    private static final Logger log = LogManager.getLogger();

    private String url;
    private HttpClient httpClient;
    private int maxConnections = 5;
    private int timeoutSeconds = 60;

    @Override
    public void afterPropertiesSet() throws Exception {
        createHttpClient();
    }

    private void createHttpClient() {
        int timeoutMillis = (int) TimeUnit.SECONDS.toMillis(timeoutSeconds);

        RequestConfig requestConfig = RequestConfig.custom()
            .setConnectionRequestTimeout(timeoutMillis)
            .setConnectTimeout(timeoutMillis)
            .setSocketTimeout(timeoutMillis)
            .build();

        httpClient = HttpClientBuilder.create()
            .setMaxConnPerRoute(maxConnections)
            .setMaxConnTotal(maxConnections)
            .setDefaultRequestConfig(requestConfig)
            .build();
    }

    /**
     * Кондуктор ожидает таймстемп по нулевому UTC.
     *
     * @param date
     * @return
     */
    private String convertToTimestamp(Date date) {
        DateTime dt = new DateTime(date.getTime());
        DateTime utcDt = dt.withZone(DateTimeZone.UTC);
        long timestamp = utcDt.toLocalDateTime().toDate().getTime();
        return Long.toString(timestamp / 1000);
    }

    /**
     * Получение выкладок тасок с возможностью применения фильтров по веткам и дате выкладки
     * Важно:
     * - даты startTime, endTime включительно
     * - кондуктор отдаёт точки в порядке убывания даты, а мы возвращаем в порядке возрастания.
     * <p>
     * curl c.test.yandex-team.ru/api/tasks_filter?project=&limit=&start_time=&end_time=
     */
    public Collection<ConductorTicket> tasksFilter(Date startTime, Date endTime, String project, Integer limit)
        throws IOException, URISyntaxException {
        URIBuilder request = new URIBuilder(url + "/api/tasks_filter?");
        request.addParameter("project", project);
        if (startTime != null) {
            request.addParameter("start_time", convertToTimestamp(startTime));
        }
        if (endTime != null) {
            request.addParameter("end_time", convertToTimestamp(endTime));
        }
        if (limit != null) {
            request.addParameter("limit", limit.toString());
        }
        HttpGet get = new HttpGet(request.build());
        String result = executeRequest(get);

        GsonBuilder gsonBuilder = new GsonBuilder()
            .setFieldNamingPolicy(FieldNamingPolicy.LOWER_CASE_WITH_UNDERSCORES)
            .setDateFormat("yyyy-MM-dd HH:mm:ss Z")
            .setPrettyPrinting();
        Gson gson = gsonBuilder.create();
        Type collectionType = new TypeToken<HashSet<ConductorTicket>>() {
        }.getType();
        Set<ConductorTicket> uniqTickets = gson.fromJson(result, collectionType);
        TreeSet<ConductorTicket> ts = new TreeSet<>(new Comparator<ConductorTicket>() {

            @Override
            public int compare(ConductorTicket o1, ConductorTicket o2) {
                int result1 = o1.getDoneAt().compareTo(o2.getDoneAt());
                return result1 == 0 ? 1 : result1;
            }
        });

        ts.addAll(uniqTickets);
        return ts;
    }

    private String executeRequest(HttpRequestBase request) throws IOException {
        try {
            log.debug(">>> Conductor request " + request.getURI());
            HttpResponse response = httpClient.execute(request);
            int httpCode = response.getStatusLine().getStatusCode();
            if (httpCode < 200 || httpCode >= 300) {
                String message = IOUtils.toString(response.getEntity().getContent());
                throw new IOException(String.format("Wrong http code: %s, message: %s at %s",
                    httpCode, message, request.getURI()));
            }
            HttpEntity entity = response.getEntity();
            return entity != null ? EntityUtils.toString(entity) : null;
        } finally {
            request.releaseConnection();
        }
    }

    public void setUrl(String url) {
        this.url = url;
    }

    public void setMaxConnections(int maxConnections) {
        this.maxConnections = maxConnections;
    }

    public void setTimeoutSeconds(int timeoutSeconds) {
        this.timeoutSeconds = timeoutSeconds;
    }
}
