package ru.yandex.market.dashboard;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 13/05/15
 */
public class GrafanaDashboard {

    private static final boolean EDITABLE = true;
    private static final String STYLE = "light";
    private static final int MAX_SPAN = 12;
    private static final String DEFAULT_ROW_HEIGHT = "250px";

    private Long id;
    private final String title;
    private final String originalTitle;
    private final String datasource;
    private final List<DashboardRow> rows;
    private final Set<String> tags;

    public GrafanaDashboard(String title, String originalTitle, String datasource, List<DashboardRow> rows, String... tags) {
        this.title = title;
        this.originalTitle = originalTitle;
        this.datasource = datasource;
        this.rows = rows;
        this.tags = new HashSet<>(Arrays.asList(tags));
    }

    public GrafanaDashboard(String title, String originalTitle, String datasource, List<DashboardRow> rows, Set<String> tags) {
        this.title = title;
        this.originalTitle = originalTitle;
        this.datasource = datasource;
        this.rows = rows;
        this.tags = tags;
    }

    public JsonObject asJson() {
        JsonObject dashboardJson = new JsonObject();
        dashboardJson.add("style", new JsonPrimitive(STYLE));
        if (id != null) {
            dashboardJson.add("id", new JsonPrimitive(id));
        }
        dashboardJson.add("title", new JsonPrimitive(title));
        dashboardJson.add("originalTitle", new JsonPrimitive(originalTitle));
        dashboardJson.add("timezone", new JsonPrimitive("browser"));
        dashboardJson.add("editable", new JsonPrimitive(EDITABLE));
        JsonArray rowsJsonArray = new JsonArray();
        for (DashboardRow row : rows) {
            rowsJsonArray.add(getRowJson(row));
        }
        dashboardJson.add("rows", rowsJsonArray);

        return dashboardJson;

    }

    private JsonObject getRowJson(DashboardRow row) {
        JsonObject rowJson = new JsonObject();
        rowJson.add("editable", new JsonPrimitive(EDITABLE));
        rowJson.add("collapse", new JsonPrimitive(false));
        rowJson.add("height", new JsonPrimitive(DEFAULT_ROW_HEIGHT));
        JsonArray panelsJsonArray = new JsonArray();
        int span = Math.max(MAX_SPAN / row.getGraphCount(), 1);
        for (DashboardGraph graph : row.getGraphs()) {
            panelsJsonArray.add(getGraphJson(graph, span));
        }
        rowJson.add("panels", panelsJsonArray);
        return rowJson;
    }

    private JsonObject getGraphJson(DashboardGraph graph, int span) {
        JsonObject graphJson = new JsonObject();

        graphJson.add("title", new JsonPrimitive(graph.getTitle()));
        graphJson.add("type", new JsonPrimitive("graph"));
        graphJson.add("span", new JsonPrimitive(span));
        graphJson.add("nullPointMode", new JsonPrimitive("null"));

        JsonArray targetsJsonArray = new JsonArray();
        for (String metric : graph.getMetrics()) {
            JsonObject targetJson = new JsonObject();
            targetJson.add("target", new JsonPrimitive(metric));
            targetsJsonArray.add(targetJson);
        }
        graphJson.add("targets", targetsJsonArray);
        graphJson.add("datasource", new JsonPrimitive(datasource));

        return graphJson;
    }

    public String getTitle() {
        return title;
    }

    public String getOriginalTitle() {
        return originalTitle;
    }

    public Long getId() {
        return id;
    }

    public void setId(Long id) {
        this.id = id;
    }

    public Set<String> getTags() {
        return tags;
    }
}
