package ru.yandex.market.dashboard;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonParser;
import com.google.gson.JsonPrimitive;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.BasicCookieStore;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.cookie.BasicClientCookie;
import org.apache.http.util.EntityUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.market.auth.PassportClient;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.util.Date;
import java.util.concurrent.TimeUnit;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 13/05/15
 */
public class GrafanaDashboardUploader implements InitializingBean {

    private static final Logger log = LogManager.getLogger();
    private static final Charset CHARSET = Charset.forName("UTF-8");
    private static final JsonParser JSON_PARSER = new JsonParser();

    private int timeoutSeconds = 60;
    private int maxConnections = 5;
    private String uploadUrl = null;
    private String token;
    private HttpClient httpClient;
    private PassportClient passportClient;

    /**
     * Login at yandex-team for passport authorization.
     */
    private String login;
    private String password;

    private BasicCookieStore cookieStore = new BasicCookieStore();

    @Override
    public void afterPropertiesSet() throws Exception {
        createHttpClient();
    }

    private void createHttpClient() throws IOException {
        int timeoutMillis = (int) TimeUnit.SECONDS.toMillis(timeoutSeconds);

        RequestConfig requestConfig = RequestConfig.custom()
            .setConnectionRequestTimeout(timeoutMillis)
            .setConnectTimeout(timeoutMillis)
            .setSocketTimeout(timeoutMillis)
            .build();

        renewPassportCookie();

        httpClient = HttpClientBuilder.create()
            .setMaxConnPerRoute(maxConnections)
            .setMaxConnTotal(maxConnections)
            .setDefaultRequestConfig(requestConfig)
            .setDefaultCookieStore(cookieStore)
            .build();

    }

    private void renewPassportCookie() {
        try {
            String cookieDomain = new URI(uploadUrl).getHost();
            for (String cookie : passportClient.getCookiesFromPassport(login, password)) {
                String[] keyValue = cookie.split("=");
                BasicClientCookie basicClientCookie = new BasicClientCookie(keyValue[0], keyValue[1]);
                basicClientCookie.setDomain(cookieDomain);
                cookieStore.addCookie(basicClientCookie);
            }
        } catch (IOException | URISyntaxException e) {
            log.error("Can't renewPassportCookie", e);
        }
    }

    private void addAuthHeaders(HttpRequest request) {
        if (cookieStore.clearExpired(new Date())) {
            renewPassportCookie();
        }
        request.addHeader("Authorization", "Bearer " + token);
    }

    public void getAndUpload(GrafanaDashboard dashboard) throws IOException {
        try {
            GrafanaDashboard existingDashboard = get(dashboard);
            if (existingDashboard != null) {
                dashboard.setId(existingDashboard.getId());
            }
        } catch (IOException e) {
            log.warn("Can't get dashboard " + dashboard.getTitle(), e);
        }

        upload(dashboard);
    }

    public void upload(GrafanaDashboard dashboard) throws IOException {
        HttpPost post = new HttpPost(uploadUrl);
        String jsonRequestString = createRequestJson(dashboard).toString();
        log.debug(jsonRequestString);
        post.setEntity(new StringEntity(jsonRequestString, CHARSET));
        addAuthHeaders(post);
        post.addHeader("Content-type", "application/json");
        executeRequest(post);
    }

    public GrafanaDashboard get(GrafanaDashboard dashboard) throws IOException {
        HttpGet request = new HttpGet(uploadUrl + "/" + dashboard.getTitle());

        log.debug(request.getURI());
        addAuthHeaders(request);

        HttpResponse response = null;
        try {
            response = httpClient.execute(request);
            int httpCode = response.getStatusLine().getStatusCode();
            if (httpCode == HttpStatus.SC_NOT_FOUND) {
                return null;
            }
            String message = IOUtils.toString(response.getEntity().getContent());
            if (httpCode == HttpStatus.SC_OK) {
                JsonElement element = JSON_PARSER.parse(message);
                JsonElement dashboardElement = element.getAsJsonObject().get("dashboard");
                if (dashboardElement == null) {
                    throw new IOException("Can't parse dashboard json");
                }
                try {
                    Gson gson = new Gson();
                    return gson.fromJson(dashboardElement, GrafanaDashboard.class);
                } catch (JsonParseException e) {
                    throw new IOException("Can't parse dashboard json");
                }
            } else {
                if (httpCode == HttpStatus.SC_MOVED_TEMPORARILY || httpCode == HttpStatus.SC_UNAUTHORIZED) { // redirect to auth
                    renewPassportCookie();
                }
                throw new IOException("Wrong http code:" + httpCode + ", message: " + message);
            }
        } finally {
            if (response != null) {
                EntityUtils.consumeQuietly(response.getEntity());
            }
            request.releaseConnection();
        }
    }

    private JsonObject createRequestJson(GrafanaDashboard dashboard) {
        JsonObject requestJson = new JsonObject();
        requestJson.add("dashboard", dashboard.asJson());
        requestJson.add("overwrite", new JsonPrimitive(true));
        return requestJson;
    }

    private String executeRequest(HttpRequestBase request) throws IOException {
        HttpResponse response = null;
        try {
            response = httpClient.execute(request);
            int httpCode = response.getStatusLine().getStatusCode();

            String message = IOUtils.toString(response.getEntity().getContent());
            if (httpCode != HttpStatus.SC_OK && httpCode != HttpStatus.SC_CREATED) {
                if (httpCode == HttpStatus.SC_MOVED_TEMPORARILY || httpCode == HttpStatus.SC_UNAUTHORIZED) { // redirect to auth
                    renewPassportCookie();
                }
                throw new IOException("Wrong http code:" + httpCode + ", message: " + message);
            } else {
                return message;
            }
        } finally {
            if (response != null) {
                EntityUtils.consumeQuietly(response.getEntity());
            }
            request.releaseConnection();
        }
    }

    @Required
    public void setUploadUrl(String uploadUrl) {
        this.uploadUrl = uploadUrl;
    }

    public void setTimeoutSeconds(int timeoutSeconds) {
        this.timeoutSeconds = timeoutSeconds;
    }

    public void setMaxConnections(int maxConnections) {
        this.maxConnections = maxConnections;
    }

    @Required
    public void setToken(String token) {
        this.token = token;
    }

    @Required
    public void setPassportClient(PassportClient passportClient) {
        this.passportClient = passportClient;
    }

    @Required
    public void setLogin(String login) {
        this.login = login;
    }

    @Required
    public void setPassword(String password) {
        this.password = password;
    }

}
