package ru.yandex.market.health;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.mongodb.MongoClient;
import com.mongodb.MongoClientOptions;
import com.mongodb.MongoClientURI;
import com.mongodb.ReadPreference;
import com.mongodb.WriteConcern;
import com.mongodb.client.FindIterable;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.DeleteOneModel;
import com.mongodb.client.model.WriteModel;
import org.bson.BsonReader;
import org.bson.BsonWriter;
import org.bson.Document;
import org.bson.codecs.Codec;
import org.bson.codecs.DecoderContext;
import org.bson.codecs.DocumentCodec;
import org.bson.codecs.EncoderContext;
import org.bson.codecs.configuration.CodecRegistries;
import org.bson.codecs.configuration.CodecRegistry;

import java.util.ArrayList;
import java.util.List;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 10/08/15
 */
public class HealthMetaDao {
    private int fetchLimit = 100_000;

    private final MongoCollection<OutputInfo> outputCollection;

    private static final CodecRegistry CODEC_REGISTRY = CodecRegistries.fromRegistries(
        MongoClient.getDefaultCodecRegistry(),
        CodecRegistries.fromCodecs(new OutputInfoCodec())
    );

    public HealthMetaDao(
        String mongoUrl,
        String dbName,
        int connectTimeoutMillis,
        int socketTimeoutMillis,
        String replicaSet,
        boolean ssl
    ) {
        this(createDatabase(mongoUrl, dbName, connectTimeoutMillis, socketTimeoutMillis, replicaSet, ssl));
    }

    @VisibleForTesting
    public HealthMetaDao(MongoDatabase database) {
        this.outputCollection = database
            .withCodecRegistry(CODEC_REGISTRY)
            .getCollection("output", OutputInfo.class);
        this.outputCollection.createIndex(new Document("table", 1));
    }

    private static MongoDatabase createDatabase(
        String mongoUrl,
        String dbName,
        int connectTimeoutMillis,
        int socketTimeoutMillis,
        String replicaSet,
        boolean ssl
    ) {
        MongoClientOptions.Builder options = MongoClientOptions.builder()
            .writeConcern(WriteConcern.MAJORITY)
            .readPreference(ReadPreference.primary())
            .connectTimeout(connectTimeoutMillis)
            .socketTimeout(socketTimeoutMillis)
            .sslEnabled(ssl);
        if (!Strings.isNullOrEmpty(replicaSet)) {
            options.requiredReplicaSetName(replicaSet);
        }
        return new MongoClient(new MongoClientURI(mongoUrl, options)).getDatabase(dbName);
    }

    public void save(OutputInfo outputInfo) {
        outputCollection.insertOne(outputInfo);
    }

    public void save(List<OutputInfo> outputInfos) {
        if (outputInfos.isEmpty()) {
            return;
        }
        outputCollection.insertMany(outputInfos);
    }

    public void delete(List<OutputInfo> outputInfos) {
        if (outputInfos.isEmpty()) {
            return;
        }
        List<WriteModel<OutputInfo>> writeModels = new ArrayList<>();
        for (OutputInfo outputInfo : outputInfos) {
            if (outputInfo.getId() == null) {
                throw new IllegalArgumentException("OutputInfo without id: " + outputInfo);
            }
            writeModels.add(new DeleteOneModel<OutputInfo>(new Document("_id", outputInfo.getId())));
        }
        outputCollection.bulkWrite(writeModels);
    }

    public List<OutputInfo> get() {
        FindIterable<OutputInfo> iterable = outputCollection
            .find()
            .limit(fetchLimit);
        List<OutputInfo> outputInfos = new ArrayList<>();
        for (OutputInfo outputInfo : iterable) {
            outputInfos.add(outputInfo);
        }
        return outputInfos;
    }

    private static class OutputInfoCodec implements Codec<OutputInfo> {
        private static final Codec<Document> DOCUMENT_CODEC = new DocumentCodec();

        @Override
        public OutputInfo decode(BsonReader reader, DecoderContext decoderContext) {
            Document document = DOCUMENT_CODEC.decode(reader, decoderContext);
            return new OutputInfo(
                document.getObjectId("_id"),
                document.getString("table"),
                document.getInteger("startTimeSeconds"),
                document.getInteger("endTimeSeconds"),
                document.getInteger("count")
            );
        }

        @Override
        public void encode(BsonWriter writer, OutputInfo outputInfo, EncoderContext encoderContext) {
            Document document = new Document();
            document.put("table", outputInfo.getTable());
            document.put("startTimeSeconds", outputInfo.getStartTimeSeconds());
            document.put("endTimeSeconds", outputInfo.getEndTimeSeconds());
            document.put("count", outputInfo.getCount());
            DOCUMENT_CODEC.encode(writer, document, encoderContext);
        }

        @Override
        public Class<OutputInfo> getEncoderClass() {
            return OutputInfo.class;
        }
    }
}
