package ru.yandex.market.health;

import com.google.common.collect.ComparisonChain;
import com.google.common.collect.Range;
import org.bson.types.ObjectId;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 10/08/15
 */
public class OutputInfo implements Comparable<OutputInfo> {

    private final ObjectId id;
    private final String table;
    private final int startTimeSeconds;
    private final int endTimeSeconds;
    private final int count;

    public OutputInfo(ObjectId id, String table, int startTimeSeconds, int endTimeSeconds, int count) {
        this.id = id;
        this.table = table;
        this.startTimeSeconds = startTimeSeconds;
        this.endTimeSeconds = endTimeSeconds;
        this.count = count;
    }

    public OutputInfo(String table, int startTimeSeconds, int endTimeSeconds) {
        this(null, table, startTimeSeconds, endTimeSeconds, 0);
    }


    public OutputInfo(String table, int startTimeSeconds, int endTimeSeconds, int count) {
        this(null, table, startTimeSeconds, endTimeSeconds, count);
    }

    public OutputInfo(String table, long startTimeMillis, long endTimeMillis, int count) {
        this(null, table, millisToSeconds(startTimeMillis), millisToSeconds(endTimeMillis), count);
    }

    public static int millisToSeconds(long millis) {
        return (int) (millis / 1000L);
    }

    public Range<Integer> getRange() {
        return Range.closedOpen(startTimeSeconds, endTimeSeconds);
    }

    public ObjectId getId() {
        return id;
    }

    public String getTable() {
        return table;
    }

    public int getStartTimeSeconds() {
        return startTimeSeconds;
    }

    public int getEndTimeSeconds() {
        return endTimeSeconds;
    }

    public int getCount() {
        return count;
    }

    @Override
    public int compareTo(OutputInfo o) {
        return ComparisonChain.start()
            .compare(startTimeSeconds, o.startTimeSeconds)
            .compare(endTimeSeconds, o.endTimeSeconds)
            .result();
    }

    @Override
    public String toString() {
        return "OutputInfo{" +
            "id=" + id +
            ", table='" + table + '\'' +
            ", startTimeSeconds=" + startTimeSeconds +
            ", endTimeSeconds=" + endTimeSeconds +
            '}';
    }
}
