package ru.yandex.market.health;

import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 16/02/15
 */
public abstract class ProcessingQueue<E> {

    protected final ReentrantLock lock = new ReentrantLock();
    private final Condition notEmpty = lock.newCondition();

    private final HashSet<E> locked = new HashSet<>();
    protected final Set<E> queue = new LinkedHashSet<>();

    public void addAll(List<E> Es) {
        lock.lock();
        for (E E : Es) {
            add(E);
        }
        lock.unlock();
    }

    public void add(E e) {
        lock.lock();
        if (!locked.contains(e) && isUpdated(e)) {
            queue.add(e);
            notEmpty.signal();
        }
        lock.unlock();
    }

    public void returnLock(E e) {
        lock.lock();
        locked.remove(e);
        if (isUpdated(e)) {
            queue.add(e);
            notEmpty.signal();
        }
        lock.unlock();
    }

    public E maybeTakeLock() {
        if (!lock.tryLock()) {
            return null;
        }
        if (queue.isEmpty()) {
            lock.unlock();
            return null;
        }
        E e = poll();
        locked.add(e);
        lock.unlock();
        return e;
    }

    public E takeLock() throws InterruptedException {
        lock.lock();
        while (queue.isEmpty()) {
            notEmpty.await();
        }
        E e = poll();
        locked.add(e);
        lock.unlock();
        return e;
    }

    public int queueSize() {
        lock.lock();
        int size = queue.size();
        lock.unlock();
        return size;
    }

    public boolean isEmpty() {
        return queueSize() == 0;
    }

    public int lockedSize() {
        lock.lock();
        int size = locked.size();
        lock.unlock();
        return size;
    }

    protected abstract boolean isUpdated(E e);

    private E poll() {
        Iterator<E> iterator = queue.iterator();
        E E = iterator.next();
        iterator.remove();
        return E;
    }

    @Override
    public String toString() {
        lock.lock();
        String string = "{queueSize=" + queueSize() + ", processingSize=" + lockedSize() + "}";
        lock.unlock();
        return string;

    }
}

